"""Dialog for assigning attributes of Boundary Conditions coverage arcs."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path
import subprocess
import webbrowser

# 2. Third party modules
import orjson
import pandas as pd
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (
    QApplication, QDialog, QDialogButtonBox, QFileDialog, QHBoxLayout, QLabel, QScrollArea, QVBoxLayout, QWidget
)

# 3. Aquaveo modules
from xms.guipy import settings
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.dialogs.xy_series_editor import XySeriesEditor
from xms.guipy.param.param_layout import ParamLayout
from xms.guipy.param.param_qt_helper import ParamQtHelper

# 4. Local modules
from xms.srh.data.bc_data import BcData
from xms.srh.data.par import bc_data_sed_inflow
from xms.srh.floodway.xms_getter import XmsGetter
from xms.srh.gui.manning_n_dialog import ManningNDialog
from xms.srh.manning_n.manning_n_calc import get_manning_n_calc_data


class BcDialog(XmsDlg):
    """The 'Assign BC' dialog."""
    def __init__(
        self, win_cont, data, allow_structures, multi_select_warning, query=None, cov_uuid=None, sel_arc_ids=None
    ):
        """Constructor.

        Args:
            win_cont (:obj:`QWidget`): Parent Qt window
            data (:obj:`ModelControl`): SRH model control class
            allow_structures (:obj:`bool`): True if the user is allowed to assign structures
            multi_select_warning (:obj:`bool`): If True, a warning message about editing multiple polygons
                will be displayed
            query (:obj:`xms.api.dmi.Query`): XMS interprocess communication object
            cov_uuid (:obj:`str`): UUID of the Boundary Conditions coverage
            sel_arc_ids (:obj:`list[int]`): XMS feature arc ids of the selected arcs
        """
        super().__init__(win_cont, 'xms.srh.gui.bc_data_dialog')
        self.setAttribute(Qt.WA_DeleteOnClose)
        self.query = query
        self.data = data
        self.allow_structures = allow_structures
        self.bc_types = self.data.param.bc_type.objects.copy()
        self.orig_bc_types = self.bc_types.copy()
        if not self.allow_structures:
            for s in BcData.structures_list:
                self.bc_types.remove(s)
        self.data.param.bc_type.objects = self.bc_types
        self.multi_select_warning = multi_select_warning
        self.hy8_crossing_dict = {}
        self.hy8_crossing_combo_box = None
        self.setup_hy8()
        if sel_arc_ids and len(sel_arc_ids) == 1:
            self.data.exit_h.wse_manning_n = lambda: self.launch_manning_n_wse()
        self.set_data_layout()
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:SRH-2D_Boundary_Conditions'
        self.param_helper = ParamQtHelper(self)
        self.widgets = dict()

        self.cov_uuid = cov_uuid
        self.sel_arc_ids = sel_arc_ids

        flags = self.windowFlags()
        self.setWindowFlags(flags & ~Qt.WindowContextHelpButtonHint)
        self.setWindowTitle('SRH2D Assign BC')

        # Define callbacks
        self.data.sediment_inflow.import_sediment_file = lambda: self._import_sediment_file()
        self.data.sediment_inflow.edit_table = lambda: self._edit_table()

        self.setup_ui()
        self.adjustSize()
        w = max(self.size().width(), 525)
        h = max(self.size().height(), 300)
        self.resize(w, h)

        self._sediment_table_widgets = {}  # The sediment table widgets
        self._set_up_sediment_table()

    def _set_up_sediment_table(self) -> None:
        """Sets up Sediment Table."""
        # Fix the table index
        sed_df = self.data.sediment_inflow.sediment_table
        sed_df.reset_index(drop=True, inplace=True)
        sed_df.index += 1

        self._sediment_table_widgets = self._get_sediment_table_widgets()
        self._enable_sediment_table_widgets()

    def _enable_sediment_table_widgets(self) -> None:
        """Enables the sediment table widgets."""
        empty = self.data.sediment_inflow.sediment_table.equals(bc_data_sed_inflow.empty_table())
        for _param_name, widget in self._sediment_table_widgets.items():
            widget.setEnabled(not empty)

    def _get_sediment_table_widgets(self) -> dict:
        """Returns the sediment table widgets as a dict.

        Returns:
            widgets: The widgets for the sediment table.
        """
        param_names = ['sediment_table', 'edit_table', 'scale_factor_label', 'scale_factor']
        widgets = {}
        for param_name in param_names:
            if param_name in self.data.sediment_inflow.param:
                param = self.data.sediment_inflow.param[param_name]
                items = self.param_helper.param_dict.get(param)
                if 'value_widget' in items:
                    widgets[param_name] = items['value_widget']
        return widgets

    def launch_manning_n_wse(self):
        """Launch the Manning's n Channel calculator and set the results."""
        # Get data to pass into the calculator
        xms_getter = XmsGetter(self.query, self.cov_uuid, self.sel_arc_ids)
        data_list, units = get_manning_n_calc_data(self.data.exit_h)

        # Check for saved calculator inputs
        previous_input = self.data.exit_h.manning_n_calculator_inputs
        input_dict = {} if not previous_input else orjson.loads(previous_input)
        # Run the Calculator
        option = self.data.exit_h.water_surface_elevation_option
        dialog = ManningNDialog(xms_getter, option, units, data_list, self, input_dict)

        if dialog.exec() == QDialog.Accepted and dialog.exit_h_wse:
            self._set_manning_n_data(dialog)

    def setup_hy8(self):
        """Initial HY-8 setup."""
        self.data.hy8_culvert.launch_hy8 = lambda: self.launch_hy8()
        self.hy8_crossing_dict = self.data.hy8_culvert.name_guid_dict_from_hy8_file()
        hy8_guid = self.data.hy8_culvert.hy8_crossing_guid
        sel_crossing_name = 'None selected'
        for name, guid in self.hy8_crossing_dict.items():
            if hy8_guid == guid:
                sel_crossing_name = name
                break
        self.update_hy8_crossing_selector()
        self.data.hy8_culvert.crossing_selector = sel_crossing_name

    def update_hy8_crossing_selector(self):
        """Modify the list of HY-8 crossings."""
        objects = ['None selected']
        objects.extend(self.hy8_crossing_dict.keys())
        self.data.hy8_culvert.param.crossing_selector.objects = objects

    def set_data_layout(self):
        """Sets layout parameters for use with ParamQtHelper."""
        self.data.param_layout = bc_data_layout()
        self.data.inlet_q.param_layout = inlet_q_layout(self)
        self.data.exit_h.param_layout = exit_h_layout(self)
        self.data.exit_q.param_layout = exit_q_layout(self)
        self.data.inlet_sc.param_layout = inlet_sc_layout(self)
        self.data.inlet_sc.water_surface_elevation.param_layout = inlet_sc_wse_layout(self)
        self.data.internal_sink.param_layout = internal_sink_layout(self)
        self.data.link.param_layout = link_layout(self)
        sed_group = 'sediment_inflow'
        sed_hlay = 'sed_table'
        self.data.sediment_inflow.param_layout = {
            'sediment_file': ParamLayout(group_id=sed_group, string_is_file_selector=True),
            'import_sediment_file': ParamLayout(group_id=sed_group),
            'imported_sediment_file': ParamLayout(group_id=sed_group, show_string=False),
            'sediment_table': ParamLayout(group_id=sed_group, dataframe_rows_spin_box=False),
            'edit_table': ParamLayout(group_id=sed_group, horizontal_layout=sed_hlay),
            'scale_factor_label': ParamLayout(group_id=sed_group, string_is_label=True, horizontal_layout=sed_hlay),
            'scale_factor': ParamLayout(group_id=sed_group, horizontal_layout=sed_hlay),
            'sediment_table_file_name': ParamLayout(group_id=sed_group, show_string=False),
        }

    def setup_ui(self):
        """Initialize the dialog widgets."""
        self._set_layout('', 'top_layout', QVBoxLayout())
        if self.multi_select_warning:
            msg = 'The selected arcs are assigned different properties.\n'\
                  'The same properties will be assigned to all selected arcs.'
            self.widgets['multi_select_warning'] = QLabel(msg)
            self.widgets['multi_select_warning'].setStyleSheet('font-weight: bold; color: red')
            self.widgets['top_layout'].addWidget(self.widgets['multi_select_warning'])
        self.widgets['scroll_area'] = QScrollArea(self)
        self.widgets['top_layout'].addWidget(self.widgets['scroll_area'])
        self.widgets['scroll_area'].setWidgetResizable(True)
        self.widgets['scroll_area'].setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
        # needed for the scroll area to function properly (see stackoverflow)
        self.widgets['scroll_area_widget'] = QWidget()
        self._set_layout('scroll_area_widget', 'scroll_area_layout', QVBoxLayout())
        self.widgets['scroll_area'].setWidget(self.widgets['scroll_area_widget'])
        self.param_helper.add_params_to_layout(self.widgets['scroll_area_layout'], self.data)
        self.widgets['scroll_area_layout'].addStretch()

        self.widgets['btn_box'] = QDialogButtonBox()
        self.widgets['top_layout'].addWidget(self.widgets['btn_box'])

        # set all widget values and hide/show
        self.param_helper.do_param_widgets(None)

        # QDialogButtonBox with Ok and Cancel buttons
        self.widgets['btn_box'].setOrientation(Qt.Horizontal)
        self.widgets['btn_box'].setStandardButtons(
            QDialogButtonBox.Cancel | QDialogButtonBox.Ok | QDialogButtonBox.Help
        )
        self.widgets['btn_box'].accepted.connect(self.accept)
        self.widgets['btn_box'].rejected.connect(super().reject)
        self.widgets['btn_box'].helpRequested.connect(self.help_requested)

        # get HY-8 crossing selector combo box
        par = self.data.hy8_culvert.param.crossing_selector
        if par in self.param_helper.param_dict:
            self.hy8_crossing_combo_box = self.param_helper.param_dict[par]['value_widget']

    def _set_layout(self, parent_name, layout_name, layout):
        """Adds a layout to the parent.

        Args:
            parent_name (:obj:`str`): Name of parent widget in self.widgets or '' for self
            layout_name (:obj:`QLay`): Name of layout in parent widget
            layout (:obj:`str`): QtLayout to be used
        """
        self.widgets[layout_name] = layout
        if parent_name:
            parent = self.widgets[parent_name]
        else:
            parent = self
        if type(parent) in [QVBoxLayout, QHBoxLayout]:
            parent.addLayout(self.widgets[layout_name])
        else:
            parent.setLayout(self.widgets[layout_name])

    def _set_manning_n_data(self, dialog):
        """Set the Manning n data from the Channel Calculator dialog.

        Args:
            dialog (:obj:`ManningNDialog`): The calculator data dialog
        """
        option = dialog.data.option
        exit_h = self.data.exit_h
        if option == 'Constant' and len(dialog.exit_h_wse) > 0:
            exit_h.constant_wse = dialog.exit_h_wse[0]
            if dialog.data.units == "U.S. Customary Units":
                exit_h.constant_wse_units = "Feet"
            else:
                exit_h.constant_wse_units = "Meters"
            self.param_helper.param_dict[exit_h.param.constant_wse_units]['value_setter'](exit_h.constant_wse_units)
            self.param_helper.param_dict[exit_h.param.constant_wse]['value_setter'](str(exit_h.constant_wse))
        if option == 'Time series' and dialog.exit_h_hour:
            # This code is inaccessible.
            update_data = {'hrs': dialog.exit_h_hour, 'm or ft': dialog.exit_h_wse}
            exit_h.time_series_wse = pd.DataFrame(data=update_data, columns=exit_h.time_series_wse.columns)
            if dialog.data.units == "U.S. Customary Units":
                exit_h.time_series_wse_units = "hrs -vs- feet"
            else:
                exit_h.time_series_wse_units = "hrs -vs- meters"
            self.param_helper.param_dict[exit_h.param.time_series_wse_units]['value_setter'](
                exit_h.time_series_wse_units
            )
        if option == 'Rating curve' and dialog.exit_h_flow:
            update_data = {'vol/sec': dialog.exit_h_flow, 'WSE': dialog.exit_h_wse}
            exit_h.rating_curve = pd.DataFrame(data=update_data, columns=exit_h.rating_curve.columns)
            if dialog.data.units == "U.S. Customary Units":
                exit_h.rating_curve_units = "cfs -vs- feet"
            else:
                exit_h.rating_curve_units = "cms -vs- meters"
            self.param_helper.param_dict[exit_h.param.rating_curve_units]['value_setter'](exit_h.rating_curve_units)

        json_dict = dialog.get_dialog_data_dict()
        exit_h.manning_n_calculator_inputs = orjson.dumps(json_dict).decode()

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def do_xy_series(self, parent_class, param_name):
        """Display an XY series editor dialog.

        Args:
            parent_class (:obj:`object`): The param object owner of the XY series being edited
            param_name (:obj:`str`): Name of the XY series param object
        """
        df = getattr(parent_class, param_name)
        df.reset_index(drop=True, inplace=True)
        df.index += 1
        dialog = XySeriesEditor(data_frame=df, series_name='curve', parent=self)
        if dialog.exec() == QDialog.Accepted:
            setattr(parent_class, param_name, dialog.model.data_frame)

    def _edit_table(self) -> None:
        """Displays a dialog in which the table can be edited."""
        df = self.data.sediment_inflow.sediment_table
        dialog = XySeriesEditor(data_frame=df, series_name='curve', parent=self)
        if dialog.exec() == QDialog.Accepted:
            self.data.sediment_inflow.sediment_table = dialog.model.data_frame
            self.param_helper.set_table_model(dialog.model.data_frame, self._sediment_table_widgets['sediment_table'])

    def _import_sediment_file(self) -> None:
        """Shows the open file dialog, reads the file to a dataframe, and shows the data in the table."""
        # Get the file path
        file_path = self._run_sediment_table_file_dialog()
        if file_path is None:
            return

        # Switch to wait cursor
        QApplication.setOverrideCursor(Qt.WaitCursor)
        QApplication.processEvents()

        # Read the file
        df = _read_sediment_file(file_path)
        if df is None:
            msg = f'Could not read file: "{str(file_path)}"'
            QApplication.restoreOverrideCursor()
            message_box.message_with_ok(
                parent=self, message=msg, app_name='SMS', icon='Error', win_icon=self.windowIcon()
            )
            return

        # Set the table and update the widgets
        self.data.sediment_inflow.sediment_table = df
        self.param_helper.set_table_model(df, self._sediment_table_widgets['sediment_table'])
        self._enable_sediment_table_widgets()
        QApplication.restoreOverrideCursor()

    def _run_sediment_table_file_dialog(self) -> Path | None:
        """Shows the open file dialog and returns the selected file."""
        last = self.data.sediment_inflow.imported_sediment_file
        path = str(Path(last).parent) if last else ''
        if not os.path.exists(path):
            path = settings.get_file_browser_directory()
        rv = QFileDialog.getOpenFileName(self, 'Select File', path, 'All files (*.*)')
        if rv[0]:
            self.data.sediment_inflow.imported_sediment_file = rv[0]
        return Path(rv[0]) if rv[0] else None

    def launch_hy8(self):
        """Launches HY-8, reads the file and updates the crossing drop down list."""
        hy8 = self.data.hy8_culvert
        if not os.path.isfile(hy8.hy8_exe) or not os.path.isfile(hy8.hy8_input_file):
            msg = 'Unable to find the HY-8 executable. Please check the HY-8 path in the "File Locations" tab of the ' \
                  '"Edit -> Preferences" dialog.'
            message_box.message_with_ok(
                parent=self, message=msg, app_name='SMS', icon='Critical', win_icon=self.windowIcon()
            )
            return
        # launch hy8 and wait for it to close
        unit_val = 0 if hy8.units == 'English' else 1
        cmd = f'"{hy8.hy8_exe}" -select_culvert_with_main_dlg -NoFlow -NoTailwater ' \
              f'-Units {unit_val} CULVERT 0 DIALOG 0 GUID "0" "{hy8.hy8_input_file}"'
        proc = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE)
        proc.wait()
        # append the last modified time to the file
        mod_time = os.path.getmtime(hy8.hy8_input_file)
        with open(hy8.hy8_input_file, 'a') as f:
            f.write(f'\nLAST_MOD_TIME {mod_time}')
        old_dict = dict(self.hy8_crossing_dict)
        # read the hy8 file and populate the objects of crossing_selector
        self.hy8_crossing_dict = hy8.name_guid_dict_from_hy8_file()
        for k, _ in old_dict.items():
            if k not in self.hy8_crossing_dict:
                msg = 'Crossings have been removed from the HY-8 file. Any HY-8 culverts in the project ' \
                      'that referenced the deleted crossings are no longer valid; errors will be reported ' \
                      'when saving simulations with invalid HY-8 culverts.'
                message_box.message_with_ok(
                    parent=self, message=msg, app_name='SMS', icon='Warning', win_icon=self.windowIcon()
                )
        self.update_hy8_crossing_selector()
        self.update_hy8_crossing_combbox()

    def update_hy8_crossing_combbox(self):
        """Sets up the items in the crossing combo box."""
        widget = self.hy8_crossing_combo_box
        txt = widget.currentText()
        if txt not in self.hy8_crossing_dict:
            txt = 'None selected'
        cnt = widget.count()
        while cnt > 1:
            widget.removeItem(cnt - 1)
            cnt = cnt - 1
        widget.addItems(self.data.hy8_culvert.param.crossing_selector.objects[1:])
        widget.setCurrentText(txt)

    def _check_bc_data_line_multiselected(self):
        """Check if mulitple arcs are selected and the bc_type is 'Bc Data'.

        Returns:
            (:obj:`bool`) : True if condition is met
        """
        if self.data.bc_type == 'Bc Data' and len(self.sel_arc_ids) > 1:
            msg = 'Unable to complete operation with multiple arcs are selected.\n' \
                  'BC Type: "Bc Data" can not be assigned to multiple arcs. '
            app_name = os.environ.get('XMS_PYTHON_APP_NAME')
            message_box.message_with_ok(parent=self, message=msg, app_name=app_name, win_icon=self.windowIcon())
            return True
        return False

    def _check_bc_data_lines_values(self):
        """Make sure the values for bc_data_lines are acceptable.

        Returns:
            (:obj:`bool`) : True if there is a problem
        """
        if self.data.param.bc_data_lines.precedence > 0:
            data_lines = self.data.bc_data_lines
            labels_equal = data_lines.upstream_line_label == data_lines.downstream_line_label
            if data_lines.specify_upstream_bcdata_line and data_lines.specify_downstream_bcdata_line and labels_equal:
                msg = 'Unable to complete operation.\n' \
                      'The upstream BCDATA line must be different from the downstream BCDATA line.'
                app_name = os.environ.get('XMS_PYTHON_APP_NAME')
                message_box.message_with_ok(parent=self, message=msg, app_name=app_name, win_icon=self.windowIcon())
                return True
        return False

    def accept(self):
        """Dialog OK method."""
        if self._check_bc_data_line_multiselected():
            return
        if self._check_bc_data_lines_values():
            return
        # Update the HY-8 crossing guid.
        widget = self.hy8_crossing_combo_box
        guid = ''
        txt = widget.currentText()
        if txt in self.hy8_crossing_dict:
            guid = self.hy8_crossing_dict[txt]
        self.data.hy8_culvert.crossing_selector = 'None selected'
        self.data.hy8_culvert.hy8_crossing_guid = guid
        if self.data.bc_type not in BcData.structures_list:
            self.data.arcs.arc_id_0 = -1
            self.data.arcs.arc_id_1 = -1
        self.data.param.bc_type.objects = self.orig_bc_types
        self.data.exit_h.param.manning_n_calculator_inputs.precedence =\
            abs(self.data.exit_h.param.manning_n_calculator_inputs.precedence)
        super().accept()


def bc_data_layout():
    """Param layout information for the BcDataInletQ class."""
    param_layout = dict()
    param_layout['inlet_q'] = ParamLayout(group_id='inlet_q', group_label='Discharge options')
    param_layout['exit_h'] = ParamLayout(group_id='exit_h', group_label='Exit water surface options')
    param_layout['exit_q'] = ParamLayout(group_id='exit_q', group_label='Discharge options')
    param_layout['inlet_sc'] = ParamLayout(group_id='inlet_sc', group_label='Supercritical inlet options')
    param_layout['internal_sink'] = ParamLayout(group_id='internal_sink', group_label='Internal sink options')
    param_layout['arcs'] = ParamLayout(group_id='arcs', group_label='Structure boundaries')
    param_layout['hy8_culvert'] = ParamLayout(group_id='hy8_culvert', group_label='HY-8 culvert options')
    param_layout['culvert'] = ParamLayout(group_id='culvert', group_label='Culvert options')
    param_layout['weir'] = ParamLayout(group_id='weir', group_label='Weir options')
    param_layout['pressure'] = ParamLayout(group_id='pressure', group_label='Pressure options')
    param_layout['gate'] = ParamLayout(group_id='gate', group_label='Gate options')
    param_layout['link'] = ParamLayout(group_id='link', group_label='Link options')
    param_layout['sediment_inflow'] = ParamLayout(
        group_id='sediment_inflow',
        group_label='Sediment inflow (Ignore if not simulating sediment transport. Only applies to simulations with '
        'sediment enabled.)'
    )
    param_layout['bc_data'] = ParamLayout(group_id='bc_data', group_label='BCDATA options')
    param_layout['bc_data_lines'] = ParamLayout(group_id='bc_data_lines', group_label='General structure options')
    param_layout['flow_direction'] = ParamLayout(group_id='bc_data_lines', )
    return param_layout


def inlet_q_layout(dlog):
    """Param layout information for the BcDataInletQ class."""
    param_layout = dict()
    param_layout['constant_q'] = ParamLayout(horizontal_layout='inlet_q_constant_q', )
    param_layout['constant_q_units'] = ParamLayout(horizontal_layout='inlet_q_constant_q', )
    param_layout['time_series_q'] = ParamLayout(
        horizontal_layout='inlet_q_time_series_q',
        xy_series_action=(dlog, 'do_xy_series'),
    )
    param_layout['time_series_q_units'] = ParamLayout(horizontal_layout='inlet_q_time_series_q', )
    return param_layout


def exit_h_layout(dlog):
    """Param layout information for the BcDataExitH class."""
    param_layout = dict()
    param_layout['constant_wse'] = ParamLayout(horizontal_layout='exit_h_constant_wse', )
    param_layout['constant_wse_units'] = ParamLayout(horizontal_layout='exit_h_constant_wse', )
    param_layout['time_series_wse'] = ParamLayout(
        xy_series_action=(dlog, 'do_xy_series'),
        horizontal_layout='exit_h_time_series_wse',
    )
    param_layout['time_series_wse_units'] = ParamLayout(horizontal_layout='exit_h_time_series_wse', )
    param_layout['rating_curve'] = ParamLayout(
        xy_series_action=(dlog, 'do_xy_series'),
        horizontal_layout='exit_h_rating_curve',
    )
    param_layout['rating_curve_units'] = ParamLayout(horizontal_layout='exit_h_rating_curve', )
    return param_layout


def exit_q_layout(dlog):
    """Param layout information for the BcDataExitQ class."""
    param_layout = dict()
    param_layout['constant_q'] = ParamLayout(horizontal_layout='exit_q_constant_q', )
    param_layout['constant_q_units'] = ParamLayout(horizontal_layout='exit_q_constant_q', )
    param_layout['time_series_q'] = ParamLayout(
        xy_series_action=(dlog, 'do_xy_series'),
        horizontal_layout='exit_q_time_series_q',
    )
    param_layout['time_series_q_units'] = ParamLayout(horizontal_layout='exit_q_time_series_q', )
    return param_layout


def inlet_sc_layout(dlog):
    """Param layout information for the BcDataExitQ class."""
    param_layout = dict()
    param_layout['constant_q'] = ParamLayout(horizontal_layout='inlet_sc_constant_q', )
    param_layout['constant_q_units'] = ParamLayout(
        horizontal_layout='inlet_sc_constant_q',
        string_is_label=True,
    )
    param_layout['constant_wse'] = ParamLayout(horizontal_layout='inlet_sc_constant_wse', )
    param_layout['constant_wse_units'] = ParamLayout(
        horizontal_layout='inlet_sc_constant_wse',
        string_is_label=True,
    )
    param_layout['time_series_q'] = ParamLayout(
        xy_series_action=(dlog, 'do_xy_series'),
        horizontal_layout='inlet_sc_time_series_q',
    )
    param_layout['time_series_q_units'] = ParamLayout(
        horizontal_layout='inlet_sc_time_series_q',
        string_is_label=True,
    )
    return param_layout


def inlet_sc_wse_layout(dlog):
    """Param layout information for the BcDataExitQ class."""
    param_layout = dict()
    param_layout['time_series_wse'] = ParamLayout(
        xy_series_action=(dlog, 'do_xy_series'),
        horizontal_layout='inlet_sc_wse_time_series_wse',
    )
    param_layout['time_series_wse_units'] = ParamLayout(
        horizontal_layout='inlet_sc_wse_time_series_wse',
        string_is_label=True,
    )
    param_layout['rating_curve'] = ParamLayout(
        xy_series_action=(dlog, 'do_xy_series'),
        horizontal_layout='inlet_sc_wse_rating_curve',
    )
    param_layout['rating_curve_units'] = ParamLayout(
        horizontal_layout='inlet_sc_wse_rating_curve',
        string_is_label=True,
    )
    return param_layout


def internal_sink_layout(dlog):
    """Param layout information for the BcDataExitQ class."""
    param_layout = dict()
    param_layout['constant_q'] = ParamLayout(horizontal_layout='internal_sink_constant_q', )
    param_layout['constant_q_units'] = ParamLayout(horizontal_layout='internal_sink_constant_q', )
    param_layout['time_series_q'] = ParamLayout(
        xy_series_action=(dlog, 'do_xy_series'),
        horizontal_layout='internal_sink_time_series_q',
    )
    param_layout['time_series_q_units'] = ParamLayout(horizontal_layout='internal_sink_time_series_q', )
    param_layout['rating_curve'] = ParamLayout(
        xy_series_action=(dlog, 'do_xy_series'),
        horizontal_layout='internal_sink_rating_curve',
    )
    param_layout['rating_curve_units'] = ParamLayout(horizontal_layout='internal_sink_rating_curve', )
    return param_layout


def link_layout(dlog):
    """Param layout information for the BcDataExitQ class."""
    param_layout = dict()
    param_layout['constant_q'] = ParamLayout(horizontal_layout='link_constant_q', )
    param_layout['constant_q_units'] = ParamLayout(horizontal_layout='link_constant_q', )
    param_layout['time_series_q'] = ParamLayout(
        xy_series_action=(dlog, 'do_xy_series'),
        horizontal_layout='link_time_series_q',
    )
    param_layout['time_series_q_units'] = ParamLayout(horizontal_layout='link_time_series_q', )
    param_layout['rating_curve'] = ParamLayout(
        xy_series_action=(dlog, 'do_xy_series'),
        horizontal_layout='link_rating_curve',
    )
    param_layout['rating_curve_units'] = ParamLayout(horizontal_layout='link_rating_curve', )
    param_layout['weir'] = ParamLayout(group_id='link_weir', group_label='Weir parameters')
    return param_layout


def _read_sediment_file(file_path: str | Path) -> pd.DataFrame | None:
    """Reads a sediment file and returns a dataframe.

    File can be either a monitor line output file (.dat), or a file containing tabular sediment data.

    Args:
        file_path: The file path.

    Returns:
        df (DataFrame): Post processed data frame with filtered columns
    """
    df = None
    try:
        # Check if the file is a monitor line file, and if not, get the number of commented lines to skip
        monitor_line_file = False
        inlet_q_file = False
        skip_row_count = 0
        with open(file_path, 'r') as file:
            for line in file:
                if line.startswith('//'):
                    skip_row_count += 1
                    if line.startswith("// Info at Inlet_Q"):
                        # Check if it's an Inlet Q file, if true, then delete last 2 columns
                        inlet_q_file = True
                else:
                    # if it's a monitor line file, it would read the headers
                    words = line.split()
                    if words[0] == 'Time(hr)':
                        monitor_line_file = True
                    break

        # Read the file
        if monitor_line_file:
            df = pd.read_csv(file_path, sep=r'\s+')
            columns_to_drop = [x for x in df.columns if not x.startswith('Qs') and not x.startswith('Time')]
            df.drop(columns=columns_to_drop, inplace=True)
        else:
            df = pd.read_csv(file_path, sep=r'\s+', skiprows=skip_row_count, header=None)

        if len(df) == 0 or len(df.columns) < 2:
            return None

        # Hardwire the table column headings
        new_names = {column_name: 'Time(hr)' if i == 0 else f'Qs{i}(cfs)' for i, column_name in enumerate(df.columns)}
        df.rename(columns=new_names, inplace=True)

        if inlet_q_file:
            # drop last two columns if it's an inlet q file
            df = df.drop(df.columns[-2:], axis=1)

        # Set the index, make all values positive, and set the first time to 0.0
        df.index += 1
        df = df.abs()
        df.loc[1, 'Time(hr)'] = 0.0
    except Exception:
        pass
    return df
