"""Dialog for specifying inputs to the 'Generate Floodway Simulation' tool."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QComboBox, QDialogButtonBox, QGridLayout, QLabel, QLineEdit, QVBoxLayout

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.api.tree import tree_util
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules


def is_srh_materials_if_coverage(item):
    """Check if a tree item is an SRH-2D Materials coverage, but only if it is a coverage.

    Args:
        item (:obj:`TreeNode`): The item to check

    Returns:
        (:obj:`bool`): True if the tree item is a 3D Bridge coverage type or is not a coverage.
    """
    if type(item.data) is xmd.CoverageItem:
        if item.model_name == 'SRH-2D' and item.coverage_type == 'Materials':
            return True
        return False
    return True


class GenerateFloodwaySimulationDialog(XmsDlg):
    """Dialog for specifying inputs to the 'Generate Floodway Simulation' tool."""
    def __init__(self, win_cont, pe_tree):
        """Initializes the class, sets up the ui, and writes the model control values.

        Args:
            win_cont (:obj:`QWidget`): Parent window
            pe_tree (:obj:`TreeNode`): The current project tree
        """
        super().__init__(win_cont, 'xms.srh.gui.generate_floodway_simulation_dialog')
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:Floodway_Delineation'
        self.line_edit = None
        self.units_cbx = None
        self.coverage_cbx = None
        self.pe_tree = pe_tree
        self._setup_ui()

    def _build_coverage_list(self, tree_node, paths):
        """Build a list of selectable SRH-2D Materials coverages to link to the new simulation.

        Returns:
            (:obj:`list[str]`): Tree item paths of the selectable items
        """
        if not tree_node:
            return

        for item in tree_node.children:
            if not item.is_ptr_item and item.model_name == 'SRH-2D' and item.coverage_type == 'Materials':
                paths.append(tree_util.build_tree_path(self.pe_tree, item.uuid))
            elif item:
                self._build_coverage_list(item, paths)

    def _setup_ui(self):
        """Initialize dialog widgets."""
        tree_util.filter_project_explorer(self.pe_tree, is_srh_materials_if_coverage)

        self.setWindowTitle('SRH2D Generate Floodway Simulation')
        main_layout = QVBoxLayout(self)
        widget_layout = QGridLayout()
        widget_layout.addWidget(QLabel('WSE Maximum Rise:'), 0, 0)
        widget_layout.addWidget(QLabel('Materials coverage:'), 1, 0)

        self.line_edit = QLineEdit('1.0')
        dbl_valid = QxDoubleValidator(bottom=0., top=100., decimals=2, parent=self)
        self.line_edit.setValidator(dbl_valid)
        widget_layout.addWidget(self.line_edit, 0, 1)

        self.units_cbx = QComboBox()
        self.units_cbx.addItems(['Feet', 'Meters'])
        widget_layout.addWidget(self.units_cbx, 0, 2)

        # Implemented coverage picker as a combobox. Could have gone with a tree picker here, but combobox is consistent
        # with the XMS toolbox tools.
        self.coverage_cbx = QComboBox()
        items = ['(none selected)']
        self._build_coverage_list(self.pe_tree, items)
        self.coverage_cbx.addItems(items)
        widget_layout.addWidget(self.coverage_cbx, 1, 1)

        button_box = QDialogButtonBox()
        button_box.setOrientation(Qt.Horizontal)
        button_box.setStandardButtons(QDialogButtonBox.Cancel | QDialogButtonBox.Ok)
        button_box.accepted.connect(self.accept)
        button_box.rejected.connect(self.reject)
        button_box.helpRequested.connect(self.help_requested)

        main_layout.addLayout(widget_layout)
        main_layout.addWidget(button_box)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def get_inputs(self):
        """Get the user inputs needed to run the create floodway simulation tool.

        Returns:
            (tuple(dict, str]`): The maximum rise dict, UUID of the selected Materials coverage to link to the new,
            simulation, or empty string if no Materials coverage was selected.
        """
        wse_maximum_rise = {'Feet': 1.0, 'Meters': 0.3048}
        rise = float(self.line_edit.text())
        units = self.units_cbx.currentText()
        wse_maximum_rise[units] = rise
        if units == 'Feet':
            wse_maximum_rise['Meters'] = 0.3048 * rise
        else:
            wse_maximum_rise['Feet'] = rise / 0.3048
        mat_item = tree_util.item_from_path(self.pe_tree, self.coverage_cbx.currentText())
        return wse_maximum_rise, mat_item.uuid if mat_item else ''
