"""The SRH-2D Model Control dialog."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QDialog, QDialogButtonBox, QScrollArea, QTabWidget, QVBoxLayout, QWidget

# 3. Aquaveo modules
from xms.api.tree import tree_util as tr_util
from xms.core.filesystem import filesystem as xmf
from xms.gdal.rasters import raster_utils as ru
from xms.gdal.rasters.raster_input import RasterInput
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs import treeitem_selector_datasets
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.treeitem_selector_datasets import TreeItemSelectorDatasetsDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.param.param_layout import ParamLayout
from xms.guipy.param.param_qt_helper import ParamQtHelper

# 4. Local modules
from xms.srh.components.generate_ceiling_runner import GenerateCeilingRunner


class ModelControlDialog(XmsDlg):
    """The SRH-2D Model Control dialog."""
    def __init__(self, win_cont, sim_comp, project_tree, query):
        """Initializes the class, sets up the ui, and writes the model control values.

        Args:
            win_cont (:obj:`QWidget`): Parent window
            sim_comp (:obj:`SimComponent`): SRH simulation component control class
            project_tree (:obj:`xms.guipy.tree.tree_node.TreeNode`): XMS project explorer tree
            query (:obj:`xms.api.dmi.Query`): Object for communicating with GMS
        """
        super().__init__(win_cont, 'xms.srh.gui.model_control_dialog')
        self.sim_comp = sim_comp
        self.data = sim_comp.data
        self.start_time_label = 'Start time (hours)'
        # force time step to be greater than or equal 1e-3
        self.data.hydro.time_step = max(1.0e-3, self.data.hydro.time_step)
        self.data.hydro.param.time_step.bounds = (1.0e-3, None)
        self.data.hydro.on_update_initial_condition = self._on_update_initial_condition

        # deprecated turbulence model
        adv = self.data.advanced
        turb = adv.turbulence_model
        items = adv.param.turbulence_model.objects[:2]
        for i, item in enumerate(adv.param.turbulence_model.objects):
            if item in items:
                adv.param.turbulence_model.objects[i] += ' - DEPRECATED'
        if turb in items:
            adv.turbulence_model += ' - DEPRECATED'

        # deprecated sediment options
        tp = self.data.sediment.transport_parameters
        par_adapt = tp.param.adaptation_length_bedload_mode
        items = par_adapt.objects[2:]
        for i, _ in enumerate(par_adapt.objects):
            if i > 1:
                par_adapt.objects[i] += ' - DEPRECATED'
        if tp.adaptation_length_bedload_mode in items:
            tp.adaptation_length_bedload_mode += ' - DEPRECATED'

        label = 'Unsteady - 3 iterations per time step. If unchecked - 1 iteration per time step.'
        self.data.advanced.param.unsteady_output.label = label
        label = 'Use ceiling file'
        self.data.hydro.param.use_pressure_ceiling.label = label
        label = 'Generate new ceiling file...'
        self.data.hydro.param.generate_pressure_ceiling.label = label

        self._main_file = sim_comp.main_file
        self._ceiling_file = os.path.join(os.path.dirname(self._main_file), 'ceiling.srhceiling')
        self._materials_file = os.path.join(os.path.dirname(self._main_file), 'materials.srhmat')
        self._query = query
        self._logger = logging.getLogger('xms.srh')
        self._orig_restart_file = ''
        if self.data.hydro.initial_condition == 'Restart File':
            if self.data.hydro.restart_file:
                self._orig_restart_file = os.path.join(
                    os.path.dirname(self._main_file), os.path.basename(self.data.hydro.restart_file)
                )
                if not os.path.isfile(self._orig_restart_file):
                    self._orig_restart_file = ''
                    self.data.hydro.restart_file = ''
        self._sort_sed_particle_diameters()
        self.project_tree = project_tree
        self.mesh_tree = self.trim_tree_to_mesh(project_tree)
        if self.mesh_tree:  # pragma: no cover
            self.mesh_tree.parent = None
        self.set_data_layout()
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:SRH-2D_Model_Control'
        self.param_helper = ParamQtHelper(self)
        self.widgets = dict()

        self.setWindowTitle('SRH2D Model Control')

        self.data.hydro.select_wse_dataset = lambda: self.select_wse_dataset()
        self.data.hydro.generate_pressure_ceiling = lambda: self.generate_ceiling()
        self.wse_dataset_uuid = self.data.hydro.water_surface_elevation_dataset
        self.data.advanced.select_nlcd_raster = lambda: self.select_nlcd_raster()
        self.setup_ui()
        self._on_update_initial_condition()

        self.wse_widget = None
        if self.data.hydro.param.water_surface_elevation_dataset in self.param_helper.param_dict:
            items = self.param_helper.param_dict[self.data.hydro.param.water_surface_elevation_dataset]
            self.wse_widget = items['value_widget']
            self.update_wse_widget()
        self.ceiling_widget = None
        if self.data.hydro.param.ceiling_file_label in self.param_helper.param_dict:
            items = self.param_helper.param_dict[self.data.hydro.param.ceiling_file_label]
            self.ceiling_widget = items['value_widget']
            self._check_ceiling_file()
        self.raster_label_widget = None
        if self.data.advanced.param.raster_label in self.param_helper.param_dict:
            items = self.param_helper.param_dict[self.data.advanced.param.raster_label]
            self.raster_label_widget = items['value_widget']
            self._update_raster_label()

        self.adjustSize()
        self.resize(self.size().width() * 1.5, self.size().height())

    def accept(self):
        """Copy selected restart file when dialog accepted."""
        self._copy_restart_file_to_component()
        self._sort_sed_particle_diameters()

        # deprecated turbulence model
        adv = self.data.advanced
        turb = adv.turbulence_model.replace(' - DEPRECATED', '')
        for i, item in enumerate(adv.param.turbulence_model.objects):
            adv.param.turbulence_model.objects[i] = item.replace(' - DEPRECATED', '')
        adv.turbulence_model = turb

        # deprecated sediment functionality
        tp = self.data.sediment.transport_parameters
        adapt = tp.adaptation_length_bedload_mode.replace(' - DEPRECATED', '')
        par_adapt = tp.param.adaptation_length_bedload_mode
        for i, item in enumerate(par_adapt.objects):
            par_adapt.objects[i] = item.replace(' - DEPRECATED', '')
        tp.adaptation_length_bedload_mode = adapt

        self.data.hydro.water_surface_elevation_dataset = self.wse_dataset_uuid
        self.data.hydro.param.select_wse_dataset.precedence = -1  # avoid IO issue
        self.data.hydro.param.generate_pressure_ceiling.precedence = -1  # avoid IO issue
        super().accept()

    def _on_update_initial_condition(self):
        """Called when the initial condition changes."""
        note = ' [Note: Start time ignored when "Initial condition" uses restart file]'
        obj = self.data.hydro.param.start_time
        widget = self.param_helper.param_dict[obj]['widget_list'][0]
        label = self.data.hydro.param.start_time.label + ':'
        if self.data.hydro.initial_condition == 'Restart File':
            label += note
        widget.setText(label)
        obj = self.data.hydro.param.restart_file
        widget = self.param_helper.param_dict[obj]['widget_list'][0]
        label = 'Restart file:' + note
        widget.setText(label)

    def _sort_sed_particle_diameters(self):
        """Sort the sediment particle diameters."""
        # sort the particle diameters
        df = self.data.sediment.particle_diameter_threshold
        if len(df) > 0:
            df.sort_values(by=['Particle Diameter Threshold (mm)'], inplace=True)
            df.reset_index(drop=True, inplace=True)

    def _copy_restart_file_to_component(self):
        """Copy a restart file to the simulation component data directory."""
        if self._orig_restart_file and self.data.hydro.initial_condition != 'Restart File':
            xmf.removefile(self._orig_restart_file)
            self.data.hydro.restart_file = ''
            return

        rst_file = self.data.hydro.restart_file
        if not rst_file:
            return
        if os.path.dirname(rst_file) == '':
            rst_file = os.path.join(os.path.dirname(self._main_file), rst_file)
        if not os.path.isfile(rst_file) or self.data.hydro.initial_condition != 'Restart File':
            self.data.hydro.restart_file = ''
            return

        if rst_file == self._orig_restart_file:
            return

        xmf.removefile(self._orig_restart_file)
        new_file = os.path.join(os.path.dirname(self._main_file), os.path.basename(self.data.hydro.restart_file))
        xmf.copyfile(rst_file, new_file)
        self.data.hydro.restart_file = os.path.basename(new_file)

    def trim_tree_to_mesh(self, project_tree):
        """Trims the project_tree down to the mesh that is part of this simulation.

        Args:
            project_tree (:obj:`xms.guipy.tree.tree_node.TreeNode`): XMS project explorer tree

        Returns:
            (:obj:`xms.guipy.tree.tree_node.TreeNode`): The trimmed project explorer tree
        """
        if project_tree is None:
            return None
        if self.data.mesh_uuid:
            return tr_util.find_tree_node_by_uuid(project_tree, self.data.mesh_uuid)
        return None

    def set_data_layout(self):
        """Sets layout parameters for use with ParamQtHelper."""
        self.data.hydro.param_layout = hydro_layout(self)
        self.data.output.param_layout = output_layout()
        self.data.advanced.param_layout = advanced_layout(self)
        sed = self.data.sediment
        sed.param_layout = sediment_layout()
        sed.transport_parameters.param_layout = sed_transport_parameters_layout()
        sed.transport_equation_parameters.param_layout = sed_transport_equation_parameters_layout()
        sed.cohesive.param_layout = sed_cohesive_layout()

    def setup_ui(self):
        """Setup the dialog widgets."""
        # Dialog QVBoxLayout with QTabWidget then QDialogButtonBox
        self._set_layout('', 'top_layout', QVBoxLayout())
        self.widgets['tab_widget'] = QTabWidget()
        self.widgets['top_layout'].addWidget(self.widgets['tab_widget'])
        self.widgets['btn_box'] = QDialogButtonBox()
        self.widgets['top_layout'].addWidget(self.widgets['btn_box'])

        #  QTabWidget with General, Output, and Advanced tab widgets
        self.widgets['general_tab'] = QWidget()
        self.widgets['tab_widget'].addTab(self.widgets['general_tab'], 'General')
        self.widgets['output_tab'] = QWidget()
        self.widgets['tab_widget'].addTab(self.widgets['output_tab'], 'Output')
        self.widgets['advanced_tab'] = QWidget()
        self.widgets['tab_widget'].addTab(self.widgets['advanced_tab'], 'Advanced')
        self.widgets['sediment_tab'] = QWidget()
        self.widgets['tab_widget'].addTab(self.widgets['sediment_tab'], 'Sediment')

        self._setup_ui_general()
        self._setup_ui_output()
        self._setup_ui_advanced()
        self._setup_ui_sediment()

        # set all widget values and hide/show
        self.param_helper.do_param_widgets(None)

        # QDialogButtonBox with Ok and Cancel buttons
        self.widgets['btn_box'].setOrientation(Qt.Horizontal)
        self.widgets['btn_box'].setStandardButtons(
            QDialogButtonBox.Cancel | QDialogButtonBox.Ok | QDialogButtonBox.Help
        )
        self.widgets['btn_box'].accepted.connect(self.accept)
        self.widgets['btn_box'].rejected.connect(super().reject)
        self.widgets['btn_box'].helpRequested.connect(self.help_requested)

    def _setup_ui_general(self):
        """Setup the general widgets."""
        self._set_layout('general_tab', 'gentab_layout', QVBoxLayout())
        self.param_helper.add_params_to_layout(self.widgets['gentab_layout'], self.data.hydro)
        self.widgets['gentab_layout'].addStretch()

    def _setup_ui_output(self):
        """Setup the output widgets."""
        self._set_layout('output_tab', 'output_tab_layout', QVBoxLayout())
        self.param_helper.add_params_to_layout(self.widgets['output_tab_layout'], self.data.output)
        self.widgets['output_tab_layout'].addStretch()

    def _setup_ui_advanced(self):
        """Setup the advanced widgets."""
        self._set_layout('advanced_tab', 'advanced_tab_layout', QVBoxLayout())
        self.param_helper.add_params_to_layout(self.widgets['advanced_tab_layout'], self.data.advanced)
        self.widgets['advanced_tab_layout'].addStretch()

    def _setup_ui_sediment(self):
        """Setup the sediment widgets."""
        self._set_layout('sediment_tab', 'sediment_tab_layout', QVBoxLayout())
        self.param_helper.add_param(
            layout=self.widgets['sediment_tab_layout'],
            param_obj=self.data.param.enable_sediment,
            param_name='enable_sediment',
            parent_class=self.data
        )
        # add scroll area for sediment
        self.widgets['sediment_scroll'] = QScrollArea(self)
        self.widgets['sediment_tab_layout'].addWidget(self.widgets['sediment_scroll'])
        self.widgets['sediment_scroll'].setWidgetResizable(True)
        self.widgets['sediment_scroll'].setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
        # needed for the scroll area to function properly (see stackoverflow)
        self.widgets['sediment_scroll_widget'] = QWidget()
        self._set_layout('sediment_scroll_widget', 'sediment_scroll_layout', QVBoxLayout())
        self.widgets['sediment_scroll'].setWidget(self.widgets['sediment_scroll_widget'])
        self.param_helper.add_params_to_layout(self.widgets['sediment_scroll_layout'], self.data.sediment)

    def _set_layout(self, parent_name, layout_name, layout):
        """Adds a layout to the parent.

        Args:
            parent_name (:obj:`str`): Name of parent widget in self.widgets or '' for self
            layout_name (:obj:`QLay`): Name of layout in parent widget
            layout (:obj:`str`): QtLayout to be used
        """
        self.widgets[layout_name] = layout
        if parent_name:
            parent = self.widgets[parent_name]
        else:
            parent = self
        parent.setLayout(self.widgets[layout_name])

    def update_wse_widget(self, dummy=''):
        """Set the text of the WSE dataset selector widget."""
        if self.wse_widget:
            uuid_str = self.wse_dataset_uuid
            uuid, ts_idx = treeitem_selector_datasets.uuid_and_time_step_index_from_string(uuid_str)
            txt = tr_util.build_tree_path(self.mesh_tree, uuid)
            if not txt:
                self.data.hydro.water_surface_elevation_dataset = 'No Dataset selected'  # No valid dataset selected.
            else:
                self.data.hydro.water_surface_elevation_dataset = f'{txt}:{ts_idx + 1}'
            self.wse_widget.setText(self.data.hydro.water_surface_elevation_dataset)

    def select_wse_dataset(self):
        """Select the WSE dataset for initial condition."""
        if not self.mesh_tree:
            msg = 'Error: unable to get mesh from SMS.'
            app_name = os.environ.get('XMS_PYTHON_APP_NAME')
            message_box.message_with_ok(parent=self, message=msg, app_name=app_name, win_icon=self.windowIcon())
            return

        uuid_and_ts = self.data.hydro.water_surface_elevation_dataset
        uuid, ts_idx = treeitem_selector_datasets.uuid_and_time_step_index_from_string(uuid_and_ts)
        dialog = TreeItemSelectorDatasetsDlg(
            title='Select Dataset',
            pe_tree=self.mesh_tree,
            selected_dataset=uuid,
            selected_time_step=ts_idx,
            query=self._query,
            show_root=True,
            parent=self
        )
        if dialog.exec() == QDialog.DialogCode.Accepted:
            dset_uuid = dialog.get_selected_item_uuid()
            if dset_uuid:
                dset_item = tr_util.find_tree_node_by_uuid(self.project_tree, dset_uuid)
                if dset_item is not None and dset_item.data_location != 'NODE':
                    msg = 'Error: Invalid dataset selected. The dataset must have values at points/nodes ' \
                          '(not cells/elements). Select a dataset with values at points/nodes.'
                    app_name = os.environ.get('XMS_PYTHON_APP_NAME')
                    message_box.message_with_ok(parent=self, message=msg, app_name=app_name, win_icon=self.windowIcon())
                    return
                ts = dialog.get_selected_time_step_string()
                uuid_and_ts = f'{dset_uuid}{ts}'  # Store a string with both the uuid and the time step index
                self.wse_dataset_uuid = uuid_and_ts
                self.data.hydro.water_surface_elevation_dataset = tr_util.build_tree_path(self.mesh_tree, uuid)
                self.update_wse_widget()

    def select_nlcd_raster(self):
        """Creates SRH material file from nlcd raster."""
        # select the raster
        old_uuid = self.data.advanced.raster_uuid
        self.data.advanced.raster_uuid = ''
        dlg = TreeItemSelectorDlg(
            title='Select NLCD Raster',
            parent=self,
            pe_tree=self.project_tree,
            target_type=None,
            selectable_xms_types=['TI_IMAGE'],
            previous_selection=old_uuid
        )
        if dlg.exec() == QDialog.DialogCode.Accepted and dlg.get_selected_item_uuid():
            show_msg = False
            raster_uuid = dlg.get_selected_item_uuid()
            raster_file = self._query.item_with_uuid(raster_uuid)
            if raster_file:
                ri = RasterInput(raster_file)
                if not ru.is_index_raster(ri):
                    show_msg = True
                else:
                    self.data.advanced.raster_uuid = raster_uuid
            else:
                show_msg = True
            if show_msg:
                msg = 'Error: The selected raster is not an index raster.'
                message_box.message_with_ok(parent=self, message=msg, app_name='SRH-2D', win_icon=self.windowIcon())

        self._update_raster_label()

    def _update_raster_label(self):
        """Update the raster label."""
        label = 'No raster selected'
        raster = tr_util.find_tree_node_by_uuid(self.project_tree, self.data.advanced.raster_uuid)
        if raster:
            label = raster.name
        if self.raster_label_widget:
            self.raster_label_widget.setText(label)

    def generate_ceiling(self):
        """Creates SRH Ceiling file from 3D bridge definitions."""
        # make sure that there is a mesh in this simulation
        app_name = os.environ.get('XMS_PYTHON_APP_NAME')
        if not self.mesh_tree:
            msg = 'Error: unable to get mesh from this simulation.'
            message_box.message_with_ok(parent=self, message=msg, app_name=app_name, win_icon=self.windowIcon())
            self._logger.error(msg)
            return
        xmf.removefile(self._ceiling_file)
        runner = GenerateCeilingRunner(self.sim_comp, self._query, self)
        if not runner.generate_ceiling():
            if len(runner.error_msg) > 0:
                message_box.message_with_ok(
                    parent=self, message=runner.error_msg, app_name=app_name, win_icon=self.windowIcon()
                )
        self._check_ceiling_file()

    def _check_ceiling_file(self):
        """Checks if the ceiling file exists and updates the variable."""
        if os.path.isfile(self._ceiling_file):
            self.data.hydro.ceiling_file_label = 'Ceiling file defined with simulation'
        else:
            self.data.hydro.ceiling_file_label = 'No ceiling file defined'
        if self.ceiling_widget:
            self.ceiling_widget.setText(self.data.hydro.ceiling_file_label)

    def help_requested(self):  # pragma: no cover
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)


def hydro_layout(dlg):
    """Param layout for hydro fields."""
    param_layout = dict()

    group_id = 'initial_condition'
    group_label = 'Initial condition'
    param_layout['initial_condition'] = ParamLayout(
        group_id=group_id,
        group_label=group_label,
    )
    param_layout['restart_file'] = ParamLayout(
        group_id=group_id,
        group_label=group_label,
        string_is_file_selector=True,
    )
    param_layout['initial_water_surface_elevation'] = ParamLayout(
        group_id=group_id,
        group_label=group_label,
        horizontal_layout='initial_water_surface_elevation',
    )
    param_layout['initial_water_surface_elevation_units'] = ParamLayout(
        group_id=group_id,
        group_label=group_label,
        horizontal_layout='initial_water_surface_elevation',
    )
    param_layout['select_wse_dataset'] = ParamLayout(
        group_id=group_id, group_label=group_label, horizontal_layout='select_wse_dataset'
    )
    param_layout['water_surface_elevation_dataset'] = ParamLayout(
        group_id=group_id, group_label=group_label, string_is_label=True, horizontal_layout='select_wse_dataset'
    )

    param_layout['generate_pressure_ceiling'] = ParamLayout(horizontal_layout='generate_pressure_ceiling')
    param_layout['ceiling_file_label'] = ParamLayout(
        string_is_label=True, horizontal_layout='generate_pressure_ceiling'
    )
    return param_layout


def output_layout():
    """Param layout for output fields."""
    param_layout = dict()
    param_layout['output_frequency'] = ParamLayout(horizontal_layout='output_frequency')
    param_layout['output_frequency_units'] = ParamLayout(horizontal_layout='output_frequency')
    group_label = 'Maximum datasets'
    group_id = 'max_datasets'
    param_layout['output_maximum_datasets'] = ParamLayout(
        group_id=group_id,
        group_label=group_label,
    )
    param_layout['specify_maximum_dataset_time_range'] = ParamLayout(
        group_id=group_id,
        group_label=group_label,
    )
    param_layout['maximum_dataset_minimum_time'] = ParamLayout(
        group_id=group_id,
        group_label=group_label,
    )
    param_layout['maximum_dataset_maximum_time'] = ParamLayout(
        group_id=group_id,
        group_label=group_label,
    )
    return param_layout


def advanced_layout(dlg):
    """Param layout for advanced fields."""
    param_layout = dict()
    group_label = 'Secondary time step'
    group_id = 'secondary_ts'
    param_layout['use_secondary_time_step'] = ParamLayout(
        group_id=group_id,
        group_label=group_label,
    )
    param_layout['secondary_time_step'] = ParamLayout(
        group_id=group_id,
        group_label=group_label,
    )
    param_layout['activation_time'] = ParamLayout(
        group_id=group_id,
        group_label=group_label,
    )
    group_label = 'Termination criteria'
    group_id = 'termination_criteria'
    param_layout['use_termination_criteria'] = ParamLayout(
        group_id=group_id,
        group_label=group_label,
    )
    param_layout['netq_value'] = ParamLayout(
        group_id=group_id,
        group_label=group_label,
    )

    group_label = 'Materials from NLCD'
    group_id = 'materials_nlcd'
    param_layout['specify_materials_nlcd'] = ParamLayout(
        group_id=group_id,
        group_label=group_label,
    )
    param_layout['select_nlcd_raster'] = ParamLayout(
        horizontal_layout='select_nlcd_raster',
        group_id=group_id,
        group_label=group_label,
    )
    param_layout['raster_label'] = ParamLayout(
        string_is_label=True,
        horizontal_layout='select_nlcd_raster',
        group_id=group_id,
        group_label=group_label,
    )
    param_layout['background_manning_n'] = ParamLayout(
        group_id=group_id,
        group_label=group_label,
    )
    return param_layout


def sediment_layout():
    """Param layout for sediment fields."""
    param_layout = dict()
    param_layout['particle_diameter_threshold'] = ParamLayout(size_policy_min=True, )
    param_layout['transport_equation_parameters'] = ParamLayout(
        group_id='transport_equation_parameters',
        group_label='Transport Equation Parameters',
    )
    param_layout['transport_parameters'] = ParamLayout(
        group_id='transport_parameters',
        group_label='Non-Transport Equation Dependent Parameters',
    )
    param_layout['cohesive'] = ParamLayout(
        group_id='cohesive',
        group_label='Cohesive Sediment Modeling Options',
    )
    return param_layout


def sed_transport_parameters_layout():
    """Param layout for sediment transport parameters fields."""
    param_layout = dict()
    param_layout['deposition_coefficient'] = ParamLayout(
        group_id='deposition_coefficient',
        group_label='Adaptation Coefficients for Suspended Load',
    )
    param_layout['erosion_coefficient'] = ParamLayout(group_id='deposition_coefficient', )
    param_layout['adaptation_length_bedload_mode'] = ParamLayout(
        group_id='adaptation_length_bedload_mode',
        group_label='Adaptation Length for Bedload Transport',
    )
    param_layout['adaptation_length_bedload_length'] = ParamLayout(group_id='adaptation_length_bedload_mode', )
    param_layout['active_layer_thickness_mode'] = ParamLayout(
        group_id='active_layer_thickness_mode',
        group_label='Active Layer Thickness Specification',
    )
    param_layout['active_layer_constant_thickness'] = ParamLayout(group_id='active_layer_thickness_mode', )
    param_layout['active_layer_d90_scale'] = ParamLayout(group_id='active_layer_thickness_mode', )
    return param_layout


def sed_transport_equation_parameters_layout():
    """Param layout for sediment transport equation fields."""
    param_layout = dict()
    param_layout['lower_transport_parameters'] = ParamLayout(
        group_id='lower_transport_parameters',
        group_label='Lower Size Class Transport Equation Parameters',
    )
    param_layout['higher_transport_parameters'] = ParamLayout(
        group_id='higher_transport_parameters',
        group_label='Higher Size Class Transport Equation Parameters',
    )
    return param_layout


def sed_cohesive_layout():
    """Param layout for cohesive sediment fields."""
    param_layout = dict()
    param_layout['bulk_density'] = ParamLayout(horizontal_layout='bulk_density')
    param_layout['bulk_density_units'] = ParamLayout(horizontal_layout='bulk_density')
    param_layout['surface_erosion'] = ParamLayout(
        group_id='surface_erosion',
        group_label='Erosion Rate Parameters',
    )
    param_layout['mass_erosion'] = ParamLayout(group_id='surface_erosion', )
    param_layout['surface_erosion_constant'] = ParamLayout(group_id='surface_erosion', )
    param_layout['mass_erosion_constant'] = ParamLayout(group_id='surface_erosion', )
    param_layout['erosion_units'] = ParamLayout(group_id='surface_erosion', )
    param_layout['full_depostion'] = ParamLayout(
        group_id='full_depostion',
        group_label='Deposition Rate Parameters',
    )
    param_layout['partial_depostion'] = ParamLayout(group_id='full_depostion', )
    param_layout['equilibrium_concentration'] = ParamLayout(group_id='full_depostion', )
    param_layout['deposition_units'] = ParamLayout(group_id='full_depostion', )
    param_layout['fall_velocity_data_file'] = ParamLayout(string_is_file_selector=True, )
    param_layout['erosion_rate_data_file'] = ParamLayout(string_is_file_selector=True, )
    return param_layout
