"""Map Obstructions coverage locations and attributes to the SRH-2D domain."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules
from xms.data_objects.parameters import FilterLocation
from xms.guipy.data.target_type import TargetType

# 4. Local modules


class ObstructionsMapper:
    """Class for mapping obstructions coverage to a mesh for SRH-2D."""
    def __init__(self, coverage_mapper):
        """Constructor."""
        self._logger = logging.getLogger('xms.srh')
        self._grid_units = 'EN' if 'FOOT' in coverage_mapper.grid_units else 'SI'
        self._coverage = coverage_mapper.obstructions_coverage
        self._obstructions_component = coverage_mapper.obstructions_component
        self._obstructions_decks = None
        self._obstructions_piers = None

    def do_map(self):
        """Creates the obstructions list for export."""
        self._build_obstructions_piers()
        self._build_obstructions_decks()

    def _build_obstructions_piers(self):
        """Builds obstruction piers from the coverage."""
        points = self._coverage.get_points(FilterLocation.PT_LOC_DISJOINT)
        obs_data = self._obstructions_component.data
        for point in points:
            comp_id = self._obstructions_component.get_comp_id(TargetType.point, point.id)
            if comp_id is None or comp_id < 0:
                comp_id = 0  # pragma: no cover

            obs = obs_data.obstruction_data_param_from_id(comp_id)
            units_str = 'EN'
            if obs.units == 'Meters':
                units_str = 'SI'
            if units_str != self._grid_units:
                msg = f'The units specified for the obstruction on point id: {point.id} do not match the units of ' \
                      f'the mesh. It is recommended that the obstruction units be consistent with the mesh units.'
                self._logger.warning(msg)
            if obs.porosity == 1.0:
                msg = f'Obstruction porosity is set to 1.0 on point id: {point.id}. Typically this value is set to 0.0.'
                self._logger.warning(msg)
            pier_str = f'PierParams {point.id} {obs.width} {obs.thickness} {obs.drag} {obs.porosity} {units_str}' \
                       f' {point.x} {point.y} {point.z}'

            if not self._obstructions_piers:
                self._obstructions_piers = []

            self._obstructions_piers.append(pier_str)

    def _build_obstructions_decks(self):
        """Builds obstruction piers from the coverage."""
        obs_data = self._obstructions_component.data
        arcs = self._coverage.arcs
        for arc in arcs:
            comp_id = self._obstructions_component.get_comp_id(TargetType.arc, arc.id)
            if comp_id is None or comp_id < 0:
                comp_id = 0  # pragma: no cover

            obs = obs_data.obstruction_data_param_from_id(comp_id)
            units_str = 'EN'
            if obs.units == 'Meters':
                units_str = 'SI'
            if units_str != self._grid_units:
                msg = f'The units specified for the obstruction on arc id: {arc.id} do not match the units of the ' \
                      f'mesh. It is recommended that the obstruction units be consistent with the mesh units.'
                self._logger.warning(msg)
            if obs.porosity == 1.0:
                msg = f'Obstruction porosity is set to 1.0 on arc id: {arc.id}. Typically this value is set to 0.0.'
                self._logger.warning(msg)
            deck_str_list = [f'DeckParams {arc.id} {obs.width} {obs.thickness} {obs.drag} {obs.porosity} {units_str}']
            pt = arc.start_node
            deck_str_list.append(f' {pt.x} {pt.y} {pt.z}')
            pts = arc.vertices
            for pt in pts:
                deck_str_list.append(f' {pt.x} {pt.y} {pt.z}')
            pt = arc.end_node
            deck_str_list.append(f' {pt.x} {pt.y} {pt.z}')
            deck_str = ''.join(deck_str_list)

            if not self._obstructions_decks:
                self._obstructions_decks = []

            msg = ''
            if len(pts) > 9:
                msg = f'Invalid obstruction defined on arc id: {arc.id}. The maximum number of points that ' \
                      f'may define an obstruction arc is 11. SRH-Pre will not read more values than 11 points. ' \
                      f'Reduce the number of vertices on the arc.'

            if not msg and len(deck_str) > 400:
                msg = f'Invalid obstruction defined on arc id: {arc.id}. The definition of the obstruction ' \
                      f'is more than 400 characters. SRH-Pre will not read more than 400 characters. Reduce the ' \
                      f'number of vertices on the arc.'

            if not msg:
                self._obstructions_decks.append(deck_str)
            else:
                self._logger.error(msg)
