"""Worker thread for running SRH-2D solution post processing."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os
import sys

# 2. Third party modules
import orjson
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg

# 4. Local modules
from xms.srh.model.srh_post_process import post_process
from xms.srh.srh_run import RunSRH


class SRHPostRunner(QThread):
    """Call srh_post when an *_XMDFC.h5 file is opened in SMS."""
    processing_finished = Signal()

    def __init__(self):
        """Construct the Runner."""
        super().__init__()
        self._logger = logging.getLogger('xms.srh')
        self._node_file_name = ''

    def run(self):
        """Converts a cell-based *._XMDFC.h5 file to node-based datasets."""
        try:
            with open('srh_post.json', 'rb') as f:
                data = orjson.loads(f.read())
            self._node_file_name = post_process(data)
        except Exception as error:
            self._logger.exception(f'Error converting cell-based solution to node-based datasets: {str(error)}')
        finally:
            self.processing_finished.emit()

    def send(self):
        """Send node-base datasets to SMS."""
        query = Query()
        runner = RunSRH()
        params = [{'solution_file': self._node_file_name}]
        runner.read_solution(query, params, win_cont=None)
        query.send()


def run_post_process(json_file_path=None, win_cont=None):
    """Entry point for srh_post processing.

    Args:
        json_file_path (:obj:`str`): Path to srha-post.json file.
        win_cont (:obj:`QWidget`): Parent window
    """
    if win_cont is not None:
        run_post_process_with_feedback(win_cont)
    elif json_file_path is not None:
        # Running as a model executable. Only perform post-processing and output
        # to the Simulation Run Queue's command line output window. Loading of
        # the node-based datasets happens with a separate GUI command.
        try:
            json_file_path = json_file_path.strip()
            with open(json_file_path, 'rb') as f:
                data = orjson.loads(f.read())
            jdir = os.path.dirname(json_file_path)
            if os.getcwd() != jdir:
                os.chdir(jdir)
            post_process(data)
        except Exception:
            logging.getLogger('xms.srh').exception('Error converting cell-based solution to node-based datasets:')
            sys.exit(-1)  # Give negative return code so we go red in Run Queue


def run_post_process_with_feedback(win_cont):
    """Driver for the solution import script with a feedback dialog.

    Args:
        win_cont (:obj:`QWidget`): Parent window
    """
    worker = SRHPostRunner()
    display_text = {
        'title': 'Reading SRH-2D Cell-based Solution',
        'working_prompt': 'Processing SRH-2D cell-based solution. Please wait...',
        'error_prompt': 'Error(s) encountered processing cell-based solution. Review log output for more details.',
        'warning_prompt': 'Warning(s) encountered processing cell-based solution. Review log output for more details.',
        'success_prompt': 'Successfully converted cell-based solution to node-based datasets.',
        'note': '',
        'auto_load': 'Auto load data into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.srh', worker, win_cont)
    feedback_dlg.exec()
