"""Provide progress feedback when running the SRH-2D solution post-process in SMS."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import shlex

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules


class SRHPostTracker:
    """Class that tracks SRH-2D model run post-processing."""
    prog = None
    query = None
    echo_file = None
    echo_pos = 0
    num_times = 0.0

    @staticmethod
    def progress_function():
        """Calculates the progress and sends it to SMS."""
        if not SRHPostTracker.echo_file:
            SRHPostTracker.echo_file = SRHPostTracker.prog.command_line_output_file

        found_iter = False
        current_ts = 0
        try:
            with open(SRHPostTracker.echo_file, "r") as f:
                f.seek(SRHPostTracker.echo_pos)
                echo_line = f.readline()
                while echo_line:
                    if (echo_line.endswith('\n') or echo_line.endswith('\r')) and \
                            echo_line.strip().startswith("Processed timestep: "):
                        echo_vals = shlex.split(echo_line)
                        try:
                            current_ts = float(echo_vals[2])
                            if SRHPostTracker.num_times == 0.0:
                                SRHPostTracker.num_times = float(echo_vals[5])
                        except IndexError:
                            pass
                        found_iter = True
                        SRHPostTracker.echo_pos = f.tell()
                    echo_line = f.readline()
        except Exception:
            pass  # File might not exist yet

        if found_iter:
            percent_done = (current_ts / SRHPostTracker.num_times) * 100.0
            SRHPostTracker.query.update_progress_percent(percent_done)

    @staticmethod
    def start_tracking():
        """Entry point for the SRH-Post progress script."""
        SRHPostTracker.query = Query(progress_script=True)
        SRHPostTracker.prog = SRHPostTracker.query.xms_agent.session.progress_loop
        SRHPostTracker.prog.set_progress_function(SRHPostTracker.progress_function)
        SRHPostTracker.prog.start_loop()
