"""PolygonsFromUGridBoundaryTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.tool_core import IoDirection, Tool

# 4. Local modules
from xms.tool.algorithms.coverage.polygons_from_ugrid_boundary import polygons_from_ugrid_boundary


class PolygonsFromUGridBoundaryTool(Tool):
    """Tool to ugrid boundary to polygons."""
    ARG_INPUT_UGRID = 0
    ARG_OUTPUT_COVERAGE = 1

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Polygons from UGrid Boundary')
        self.ug = None
        self.loops = None
        self.new_cov = None
        self.cov_name = ''
        self.cov_nodes = {}

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.grid_argument(name='input_grid', description='Input grid'),
            self.coverage_argument(name='output_coverage', description='Output coverage name',
                                   io_direction=IoDirection.OUTPUT, optional=True)
        ]
        return arguments

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        self.logger.info('Retreiving input UGrid...')
        self.ug = self.get_input_grid(arguments[self.ARG_INPUT_UGRID].value)

        update_name_after = False
        coverage_name = self.cov_name = arguments[self.ARG_OUTPUT_COVERAGE].value
        if self.cov_name == '' or self.cov_name is None:  # default the coverage name to match the ugrid
            ug_path = arguments[self.ARG_INPUT_UGRID].text_value.split('/')
            if ug_path:
                coverage_name = ug_path[-1]
                update_name_after = True

        prj = None if not self.default_wkt else self.default_wkt

        try:
            self.new_cov = polygons_from_ugrid_boundary(self.ug, prj, coverage_name, self.logger)
        except ValueError as e:
            self.fail(str(e))
        else:
            if update_name_after:
                self.cov_name = coverage_name
                arguments[self.ARG_OUTPUT_COVERAGE].value = self.cov_name
            self.set_output_coverage(self.new_cov, arguments[self.ARG_OUTPUT_COVERAGE])
