"""AngleConventionTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.tool_core import IoDirection, Tool

# 4. Local modules
from ..algorithms.datasets.angle_convention import angle_convention


ARG_INPUT_DSET = 0
ARG_FROM_CONVENTION = 1
ARG_TO_CONVENTION = 2
ARG_OUTPUT_DATASET = 3


class AngleConventionTool(Tool):
    """Tool to convert an angle convention dataset to another angle convention."""

    def __init__(self, **kwargs):
        """Initializes the class."""
        super().__init__(name='Angle Convention', **kwargs)
        self._dataset_reader = None
        self._dir_builder = None

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.dataset_argument(name='input_dataset', description='Input scalar dataset'),
            self.string_argument(name='from_convention', description='Input angle convention',
                                 choices=['Cartesian', 'Meteorologic', 'Oceanographic'], value='Meteorologic'),
            self.string_argument(name='to_convention', description='Output angle convention',
                                 choices=['Cartesian', 'Meteorologic', 'Oceanographic'], value='Meteorologic'),
            self.dataset_argument(name='default_name', description='Output dataset name', value="new dataset",
                                  io_direction=IoDirection.OUTPUT),
        ]
        self.enable_arguments(arguments)
        return arguments

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}
        # Validate input datasets
        self._dataset_reader = self._validate_input_dataset(arguments[ARG_INPUT_DSET], errors)
        return errors

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        # Get input_type and output_type
        input_type = arguments[ARG_FROM_CONVENTION].text_value
        output_type = arguments[ARG_TO_CONVENTION].text_value

        # Extract the component, time step at a time
        dset_grid = self.get_input_dataset_grid(arguments[ARG_INPUT_DSET].text_value)

        # Run Algorithm
        self._dir_builder = angle_convention(arguments[ARG_OUTPUT_DATASET].text_value, self._dataset_reader, dset_grid,
                                             input_type, output_type, self.logger)

        self._add_output_datasets()

    def _add_output_datasets(self):
        """Add datasets created by the tool to be sent back to XMS."""
        # Send the dataset back to XMS
        self.set_output_dataset(self._dir_builder)


# def main():
#     """Main function, for testing."""
#     from xms.tool_gui.tool_dialog import ToolDialog
#     from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists
#
#     qapp = ensure_qapplication_exists()
#     tool = AngleConventionTool()
#     arguments = tool.initial_arguments()
#     arguments[ARG_INPUT_DSET].value = 'dir_dset'
#     arguments[ARG_OUTPUT_DATASET].value = 'dir_dset_out'
#     tool_dialog = ToolDialog(None, arguments, 'Angle Convention', tool=tool)
#     if tool_dialog.exec():
#         tool.run_tool(tool_dialog.tool_arguments)
#
#
# if __name__ == "__main__":
#     main()
