"""Convert between time units."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules

UNITS_SECONDS = 'seconds'
UNITS_MINUTES = 'minutes'
UNITS_HOURS = 'hours'
UNITS_DAYS = 'days'
UNITS_WEEKS = 'weeks'
UNITS_YEARS = 'years'
TIME_UNITS = [UNITS_SECONDS, UNITS_MINUTES, UNITS_HOURS, UNITS_DAYS, UNITS_WEEKS, UNITS_YEARS]
DAYS_PER_YEAR = 365.242198781
TO_SECONDS = {
    UNITS_SECONDS: 1.0,
    UNITS_MINUTES: 60.0,
    UNITS_HOURS: 60.0 * 60.0,
    UNITS_DAYS: 60.0 * 60.0 * 24.0,
    UNITS_WEEKS: 60.0 * 60.0 * 24.0 * 7.0,
    UNITS_YEARS: 60.0 * 60.0 * 24.0 * DAYS_PER_YEAR
}


class TimeUnitsConverter:
    """
    This class is used to convert raw times of known units into different units.
    """

    def __init__(self, from_units: Optional[str] = None, to_units: Optional[str] = None):
        """Initialize TimeUnitsConverter instance.

        Args:
            from_units (Optional[str]): The units to convert from.
            to_units (Optional[str]): The units to convert to.
        """
        self._from_units: str = UNITS_SECONDS
        self._to_units: str = UNITS_SECONDS
        self._factor = 1.0

        if from_units is not None:
            self.from_units = from_units
        if to_units is not None:
            self.to_units = to_units

    @property
    def to_units(self) -> str:
        """Get the units to convert to.

        Returns:
            (str): The units to convert to.
        """
        return self._to_units

    @to_units.setter
    def to_units(self, value: str) -> None:
        """Set the units to convert to.

        Args:
            value (str): The units to convert to.
        """
        self._to_units = value.lower()
        self._init_factor()

    @property
    def from_units(self) -> str:
        """Get the units to convert from.

        Returns:
            (str): The units to convert from.
        """
        return self._from_units

    @from_units.setter
    def from_units(self, value: str) -> None:
        """Set the units to convert from.

        Args:
            value (str): The units to convert from.
        """
        self._from_units = value.lower()
        self._init_factor()

    def convert_value(self, value: float) -> float:
        """Convert a time value from the "from units" to the "to units".

        Args:
            value (float): The value to convert.

        Returns:
            The converted value.
        """
        return value * self._factor

    def _init_factor(self):
        self._factor = TO_SECONDS[self._from_units] / TO_SECONDS[self._to_units]
