"""Dialog for cleaning shapes tool."""

# 1. Standard Python modules
from copy import deepcopy
from decimal import Decimal
from enum import Enum
import os
import pickle

# 2. Third party modules
import folium
from osgeo import osr
import pandas as pd
from PySide2.QtCore import QSize, Qt, QUrl, Signal
from PySide2.QtGui import QColor, QPalette
from PySide2.QtWidgets import (QAbstractItemView, QComboBox, QHeaderView, QStyledItemDelegate)
from rtree import index
from shapely.geometry import LineString, Polygon as shPolygon
from shapely.ops import unary_union

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.core.filesystem import filesystem as xfs
from xms.guipy.data.map_locator_web_parameters import ParametersObject, ParametersWebPage
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.widgets import widget_builder
from xms.tool_core.coverage_builder import CoverageBuilder

# 4. Local modules
from xms.tool_xms.algorithms.clean_actions import (collapse_arc, curvature_redistribution, IntersectPolygonsByTolerance,
                                                   prune_arc, snap_arcs)
import xms.tool_xms.algorithms.geom_funcs_with_shapely as gsh
from xms.tool_xms.coverages.clean_shapes_tool import (clean_shapes_arc_poly_db_file,
                                                      clean_shapes_ss_display_file, convert_latlon_geom_to_utm,
                                                      feet_to_decimal_degrees, meters_to_decimal_degrees,
                                                      needs_axis_swap)
from xms.tool_xms.gui.clean_shapes_dialog_ui import Ui_CleanShapesDialog
from xms.tool_xms.gui.clean_shapes_viewer_dialog import CleanShapesViewerDialog
from xms.tool_xms.utils.geom_drawer import clear_geometry, DisplayParameters

# Columns in the primary table
CA_DISPLAY_TYPE = 0
CA_ID = 1
CA_LENGTH = 2
CA_NUM_SEGMENTS = 3
CA_NUM_CONNECTED_ARCS = 4
CA_DIST_TO_PRIMARY = 5
CA_GEOM_LIST_IDX = 6

CP_DISPLAY_TYPE = 0
CP_ID = 1
CP_NUM_ARCS = 2
CP_AREA = 3
CP_PERIM = 4
CP_DIST_TO_PRIMARY = 5
CP_GEOM_LIST_IDX = 6

STR_DISP_TYPE = 'Display'
STR_DIST_TO_P = 'Distance to Primary'
STR_LEN = 'Length'
STR_NUM_SEGS = '# Segments'
STR_NUM_CX = '# Connected Arcs'
STR_NUM_ARCS = '# Arcs'
STR_AREA = 'Area'
STR_PERIM = 'Perimeter'
STR_GEOM_IDX = 'Geometry List Index'
STR_ID = 'ID'


class Triggers(Enum):
    """Enumeration for update triggers."""
    TE_NONE = 0
    TE_GEOM_TYPE = 1
    TE_PRIMARY_SEL = 2
    TE_SECONDARY_SEL = 3
    TE_FILTER = 4
    TE_ACTION_OPT = 5
    TE_CHK_ACTION_2 = 6
    TE_CLEAR_SECONDARY = 7
    TE_ACTION_SEL = 8
    TE_ITEM_DISPLAY = 9
    TE_ACCEPT_ACTION = 10
    TE_SET_AS_PRIMARY = 11
    TE_CREATE_PREVIEW = 12
    TE_ZOOM_TO_PRIMARY = 13
    TE_ZOOM_TO_SECONDARY = 14
    TE_DELETE_PRIMARY = 15
    TE_DELETE_SECONDARY = 16
    TE_DISPLAY_MAP = 17
    TE_FILL_POLYS = 18


# filter method
FM_BOUNDS_MULT = 0
FM_MAX_DIST = 1


def load_tool_output():
    """Load geometry data from the tool.

    Returns:
        (:obj:`tuple[pandas.DataFrame, Coverage, str]`): The results DataFrame or None on error
    """
    dfs = []
    try:
        filenames = [clean_shapes_ss_display_file(), clean_shapes_arc_poly_db_file()]
        for filename in filenames:
            # Read the pickled plot DataFrame and global log output
            with open(filename, 'rb') as f:
                data = pickle.load(f)
            xfs.removefile(filename)
            dfs.append(data['df'])
    except Exception:
        pass
    return dfs


class DisplayTypeCbxDelegate(QxCbxDelegate):
    """A combobox delegate."""
    display_type_changed = Signal(int)

    def on_index_changed(self, index):
        """Slot to close the editor when the user selects an option.

        Args:
            index (object): unused
        """
        self.display_type_changed.emit(index)


class FloatDelegate(QStyledItemDelegate):
    """A float delegate."""

    def displayText(self, text, locale):  # noqa: N802
        """Display `text` in the selected with the selected number of digits.

        Args:
            text:   string / QVariant from QTableWidget to be rendered
            locale: locale for the text
        """
        return str(Decimal(text).quantize(Decimal("0.000")))


class QxPandasTableModelCheapHeaders(QxPandasTableModel):
    """Class derived from QxPandasTableModel to avoid expensive headerData calls."""
    finished_sort = Signal(bool)

    def __init__(self, data_frame, parent, is_primary, geom_type):
        """Initializes the class.

        Args:
            data_frame (:obj:`pandas.DataFrame`): The pandas DataFrame
            parent (:obj:`QWidget`): The Qt parent
            is_primary (bool): Is the primary ss
            geom_type (str): Geometry type for the dialog ('Arc' or 'Polygon')
        """
        super().__init__(data_frame, parent)
        self._column_names = data_frame.columns.values.tolist()
        self._is_primary = is_primary
        self._set_geom_type(geom_type)

    def _set_geom_type(self, geom_type):
        """Initializes the filter model.

        Args:
            geom_type (str): Geometry type
        """
        self._geom_type = geom_type
        if self._geom_type == 'Arc':
            self._float_sort_types = [CA_LENGTH, CA_DIST_TO_PRIMARY]
            self._int_sort_types = [CA_ID, CA_NUM_CONNECTED_ARCS, CA_NUM_SEGMENTS, CA_GEOM_LIST_IDX]
            self._col_geom_list_idx = int(CA_GEOM_LIST_IDX)
            self._col_disp_type = int(CA_DISPLAY_TYPE)
            self._col_dist_to_p = int(CA_DIST_TO_PRIMARY)
            self._col_feat_id = int(CA_ID)
            self._primary_hide_list = [CA_DIST_TO_PRIMARY, CA_GEOM_LIST_IDX]
            self._secondary_hide_list = [CA_DISPLAY_TYPE, CA_GEOM_LIST_IDX]
            self._sort_col = int(CA_ID) if self._is_primary else int(CA_DIST_TO_PRIMARY)
        else:
            self._float_sort_types = [CP_AREA, CP_PERIM, CP_DIST_TO_PRIMARY]
            self._int_sort_types = [CP_ID, CP_NUM_ARCS, CP_GEOM_LIST_IDX]
            self._col_geom_list_idx = int(CP_GEOM_LIST_IDX)
            self._col_disp_type = int(CP_DISPLAY_TYPE)
            self._col_dist_to_p = int(CP_DIST_TO_PRIMARY)
            self._col_feat_id = int(CP_ID)
            self._primary_hide_list = [CP_DIST_TO_PRIMARY, CP_GEOM_LIST_IDX]
            self._secondary_hide_list = [CP_DISPLAY_TYPE, CP_GEOM_LIST_IDX]
            self._sort_col = int(CP_ID) if self._is_primary else int(CP_DIST_TO_PRIMARY)

        self._sort_order = Qt.AscendingOrder

    def sort(self, column, order=None):
        """Sorts the model by column in the given order.

        Args:
            column (int): The column to sort.
            order (QtCore.Qt.SortOrder): The sort order.
        """
        self._sort_col = column
        self._sort_order = order

        colname = self.data_frame.columns.tolist()[column]
        self.layoutAboutToBeChanged.emit()
        self.data_frame.sort_values(colname, ascending=order != Qt.DescendingOrder, inplace=True)
        self.data_frame.reset_index(inplace=True, drop=True)
        self.layoutChanged.emit()

        self.finished_sort.emit(self._is_primary)

    def update_data_frame(self, df):
        """Sorts the model by column in the given order.

        Args:
            df (int): The dataframe.
        """
        self.data_frame = df
        self.sort(self._sort_col, self._sort_order)
        self.layoutChanged.emit()

    def headerData(self, section, orientation, role=Qt.DisplayRole):  # noqa: N802
        """Returns the data for the given role and section in the header.

        Args:
            section (:obj:`int`): The section.
            orientation (:obj:`Qt.Orientation`): The orientation.
            role (:obj:`int`): The role.

        Returns:
            The data.
        """
        if role == Qt.DisplayRole:
            if orientation == Qt.Horizontal:
                return self._column_names[section]
            else:
                return section + 1  # Switch default 0-base sequential pandas Index to 1-base row number
        return super().headerData(section, orientation, role)


class CleanShapesDialog(XmsDlg):
    """Dialog for the cleaning shapes tool."""

    def __init__(self, parent, show_viewer=True, show_dlg=True, is_mpbd=False):
        """Constructor.

        Args:
            parent (:obj:`QObject`): The parent object
            show_viewer (bool): True if the map viewer should be visible (False for tests)
            show_dlg (bool): True if the dialog should be visible
            is_mpbd (bool): True if we are using the derived version for the merge polygons by distance tool
        """
        super().__init__(parent, 'xms.tool_xms.gui.clean_shapes_dialog')
        if show_dlg:
            self._super_exec = super().exec_
        self.ui = None
        self._is_mpbd_tool = is_mpbd
        self.p_model = None
        self.s_model = None
        self._allow_display_update = False
        self._in_setup = True
        self._simplify_tol = 100000

        self._map_viewer = None

        self._geom_type = 'Arc'

        self._do_auto_updates = False

        self._filter_method = FM_MAX_DIST
        self._multiplier = 1.5
        self._max_dist = 50.0

        self._show_viewer = show_viewer

        self._secondary_geoms_in_bounds = []

        self._arc_actions = ['Collapse', 'Concatenate', 'Curvature Redistribution', 'Delete', 'Prune', 'Snap']
        self._poly_actions = ['Delete', 'Merge']

        self._prune_options = ['Left', 'Right']
        self._prune_options_2 = ['mitre', 'round', 'bevel']
        self._prune_options_3 = ['out', 'in', 'none']

        # prune options
        self._prune_side = 'Left'
        self._prune_distance = 5.0
        self._join_type = 'mitre'
        self._mitre_limit = 2.5
        self._debug_arcs = 'none'
        self._num_steps = 5

        # curvature options
        self._max_delta = 30.0
        self._min_seg_length = 1.0

        # snap
        self._snap_tol = 5.0

        # merge
        self._merge_tol = 5.0

        self._collapse_options = ['Start', 'End']

        self._actions_that_use_secondary = ['Concatenate', 'Snap', 'Merge']
        self._multi_select_actions = ['Delete']

        self._sel_geom_item = None
        self._sel_geom_locs_latlon = None
        self._sel_geom_idx = None

        self._sel_geom_item_multi = None
        self._sel_geom_locs_latlon_multi = None
        self._sel_geom_idx_multi = None

        self._reset_display_colors()

        self._action_display = []
        self._primary_display = []
        self._secondary_display = []
        self._sel_secondary_display = []
        self._debug_arcs_display = []
        self._map_viewer_extents = []
        self._action_preview_extents = []
        self._sel_secondary_item_extents = None

        self._mod_geoms = None
        self._mod_latlon_locs = []
        self._mod_latlon_extents = []

        self._draw_sel_secondary_for_action_preview = True
        self._sel_secondary_geom_idx = None

        self._newest_geom_idx = None

        self._addtl_mod_geom_list_idxs = []

        self._merge_is_union = True
        self._merge_is_difference = False

        self._removed_inner_polys = []
        self._is_deleting = []

        self._iterating_for_selectable = False
        self._display_types = ['Normal', 'Simplified', 'Exclude']
        self._simplified_geom_latlon_locs = {}

        self._geom_latlon_locs = []
        self._latlon_extents = []
        self._deleted_geom_idxs = []

        self._native_wkt = None
        self._geographic_wkt = None
        self._non_geographic_wkt = None

        self._action_display_params = None
        self._map_viewer_params = None

        self._update_map_view = False
        self._draw_preview = False
        self._preview_is_displayed = False
        self._display_dashed_item = False

        self._update_trigger = Triggers.TE_NONE

        self._cur_sel_row_primary = 0
        self._cur_sel_row_secondary = 0

        self._msg_action = None
        self._msg_info = None

        temp_dir = XmEnv.xms_environ_process_temp_directory()
        self._url_action_preview_file = os.path.join(temp_dir, f'clean_shapes_action_{os.getpid()}.html')
        self._url_map_view_file = os.path.join(temp_dir, f'clean_shapes_viewer_{os.getpid()}.html')

    # def _enable_profile(self):
    #     """Enable profiling."""
    #     # pass
    #     import cProfile
    #     self.pr = cProfile.Profile()
    #     self.pr.enable()
    #
    # def _profile_report(self):
    #     """Print the profile report."""
    #     # pass
    #     import pstats
    #     import io
    #     self.pr.disable()
    #     s = io.StringIO()
    #     sortby = 'cumulative'
    #     ps = pstats.Stats(self.pr, stream=s).sort_stats(sortby)
    #     ps.print_stats()
    #     with open('c:/temp/profile.txt', 'w') as f:
    #         f.write(s.getvalue())

    def set_parameters(self, lat, long, zoom, x_min, x_max, y_min, y_max):
        """Set the parameters as a callback from the javascript web page code."""
        self._action_display_params = DisplayParameters()
        self._action_display_params.lat = lat
        self._action_display_params.long = long
        self._action_display_params.zoom = zoom
        self._action_display_params.x_min = x_min
        self._action_display_params.x_max = x_max
        self._action_display_params.y_min = y_min
        self._action_display_params.y_max = y_max

        self._map_viewer_params = self._map_viewer._display_parameters

    def _reset_display_colors(self):
        """Set default display colors."""
        self._display_colors = ['blue', 'blue', 'green', 'gray', 'yellow', 'darkred']

    def _transform_all_geom_to_latlon(self):
        """Initialize from DataFrame written by tool once we know we have read something valid."""
        if self._is_geo:
            for geom in self.tool._geometry_locs:
                new_geom_lists = [[(pt[1], pt[0]) for pt in pt_list] for pt_list in geom]
                self._geom_latlon_locs.append(new_geom_lists)
            self._latlon_extents = [self._get_transformed_bounds_from_geom(geom) for geom in self._geoms]
        else:
            for geom_idx in range(len(self._geoms)):
                geom = self._geoms[geom_idx]
                if type(geom) is LineString:
                    arc_locs = self.tool._geometry_locs[geom_idx][0]
                    if self._is_local:
                        pts_latlon = [(self._convert(loc[1], 0.0), self._convert(loc[0], 0.0)) for loc in arc_locs]
                    else:
                        pts_latlon = self._reproject_pts_to_latlon(arc_locs)

                    self._geom_latlon_locs.append([pts_latlon])
                else:  # polygon
                    outer_pts = self.tool._geometry_locs[geom_idx][0].copy()
                    len_loops = len(self.tool._geometry_locs[geom_idx])
                    holes = [self.tool._geometry_locs[geom_idx][hole_idx] for hole_idx in range(1, len_loops)]

                    if self._is_local:
                        pts_latlon = [[(self._convert(pt[1], 0.0), self._convert(pt[0], 0.0)) for pt in outer_pts]]
                        for h in holes:
                            pts_latlon.append([(self._convert(pt[1], 0.0), self._convert(pt[0], 0.0)) for pt in h])
                    else:
                        pts_latlon = [self._reproject_pts_to_latlon(outer_pts)]
                        for h in holes:
                            pts_latlon.append(self._reproject_pts_to_latlon(h))

                    self._geom_latlon_locs.append(pts_latlon)

                # get the extents
                self._latlon_extents.append(self._get_transformed_bounds_from_geom(geom))

    def _transform_geom(self, geom):
        """Transform the geometry to latlon for display.

        Args:
            geom (Object): Geometry to transform
        Return:
            geom (Object): Transformed item
            extents (list): Transformed extents
        """
        if type(geom) is LineString:
            pts = gsh.pts_from_ls(geom)
            if self._is_local:
                pts_latlon = [[(self._convert(pt[1], 0.0), self._convert(pt[0], 0.0)) for pt in pts]]
            elif self._is_geo:
                pts_latlon = [[(pt[1], pt[0]) for pt in pts]]
            else:
                pts_latlon = [self._reproject_pts_to_latlon(pts)]
        else:  # polygon
            if self._is_geo:
                pts_latlon = [[(p[1], p[0]) for p in hole.coords] for hole in geom.interiors]
                pts_latlon.insert(0, [(p[1], p[0]) for p in geom.exterior.coords])
            else:
                pts = [(p[0], p[1]) for p in geom.exterior.coords]
                holes = [[(p[0], p[1]) for p in hole.coords] for hole in geom.interiors]

                if self._is_local:
                    pts_latlon = [[(self._convert(pt[1], 0.0), self._convert(pt[0], 0.0)) for pt in h] for h in holes]
                    pts_latlon.insert(0, [(self._convert(pt[1], 0.0), self._convert(pt[0], 0.0)) for pt in pts])
                else:
                    pts_latlon = [self._reproject_pts_to_latlon(hole) for hole in holes]
                    pts_latlon.insert(0, self._reproject_pts_to_latlon(pts))

        # get the extents
        extents = self._get_transformed_bounds_from_geom(geom)
        return pts_latlon, extents

    def _get_transformed_bounds_from_geom(self, geom):
        """Get transformed extents from a list.

        Args:
            geom (Object): geometry
        Return:
           extents (list): one set of extents that covers the whole group
        """
        return self._transform_bounds(geom.bounds)

    def _transform_bounds(self, bounds):
        """Get extents from a list.

        Args:
            bounds (list): bounds to be transformed to latlon
        Return:
           extents (list): one set of extents that covers the whole group
        """
        if self._is_geo:
            return [
                [bounds[1], bounds[0]],  # (min_lat, min_lon)
                [bounds[3], bounds[2]],  # (max_lat, max_lon)
            ]
        elif self._is_local:
            pts = [[bounds[1], bounds[0]], [bounds[3], bounds[2]]]
            return [[self._convert(pt[0], 0.0), self._convert(pt[1], 0.0)] for pt in pts]
        else:
            pts = [[bounds[0], bounds[1]], [bounds[2], bounds[3]]]
            pts = self._reproject_pts_to_latlon(pts)
            return [[pts[0][0], pts[0][1]], [pts[1][0], pts[1][1]]]

    def _reproject_pts_to_latlon(self, pts, swap_output_axis=False):
        """Reproject the line locations from a projected system to geographic.

        Args:
            pts (:obj:`list`): List of the line locations [[(x,y),...],...]
            swap_output_axis (bool): reverse x and y if True
        """
        # Set up a transformation from input non-geographic to WGS 84
        source = osr.SpatialReference()
        source.ImportFromWkt(self._non_geographic_wkt)
        swap_input_axis = needs_axis_swap(source)
        target = osr.SpatialReference()
        target.ImportFromWkt(self._geographic_wkt)
        transform = osr.CoordinateTransformation(source, target)

        return [self._transform_pt(transform, point, swap_input_axis, swap_output_axis) for point in pts]

    def _convert_latlon_geom_to_utm(self, geom):
        """Reproject the line locations from a projected system to geographic.

        Args:
            geom (object): geom item to convert
        """
        non_geo_geom, _ = convert_latlon_geom_to_utm(geom, self._geographic_wkt, self._non_geographic_wkt)
        return non_geo_geom

    def _transform_pt(self, transformer, point, input_axis_swap, output_axis_swap):
        """Reproject the line locations from a projected system to geographic.

        Args:
            transformer (:obj): Transformer
            point (list): Location to reproject
            input_axis_swap (bool): reverse input x and y if True
            output_axis_swap (bool): reverse output x and y if True
        """
        # Set up a transformation from input non-geographic to WGS 84
        if input_axis_swap:
            coord = transformer.TransformPoint(point[1], point[0])
        else:
            coord = transformer.TransformPoint(point[0], point[1])
        return (coord[1], coord[0]) if output_axis_swap else (coord[0], coord[1])

    """
    UI setup
    """
    def _setup_ui(self):
        """Sets up the UI."""
        self.ui.setupUi(self)

        # turn off auto update
        self.ui.chk_auto_update.setChecked(False)
        self.ui.btn_create_preview.setEnabled(True)

        # hide delete buttons
        self.ui.btn_delete_primary.setHidden(True)
        self.ui.btn_delete_secondary.setHidden(True)

        # set up the combo boxes
        self.ui.cbx_action.addItems(self._arc_actions)

        # filter controls
        self.ui.spn_multiplier.setMinimum(0.0)
        self.ui.spn_multiplier.setMaximum(1000.0)
        self.ui.spn_multiplier.setValue(self._multiplier)
        self.ui.spn_multiplier.setSingleStep(0.5)

        validator = QxDoubleValidator()
        validator.setBottom(0.0)
        self.ui.edt_dist.setValidator(validator)
        self.ui.edt_dist.setText(str(self._max_dist))

        self.ui.rbt_distance.setChecked(True)
        self._on_change_filter()

        self._setup_model(True)
        self._setup_model(False)

        # set up primary table
        self._display_delegate = DisplayTypeCbxDelegate()
        self._display_delegate.set_strings(self._display_types)
        self.ui.tbl_primary.setItemDelegateForColumn(CA_DISPLAY_TYPE, self._display_delegate)
        self.ui.tbl_primary.setModel(self.p_model)
        self._setup_table(self.ui.tbl_primary, True)

        # set up secondary table
        self.ui.tbl_secondary.setModel(self.s_model)
        self._setup_table(self.ui.tbl_secondary, False)

        # set up the action preview window
        self._setup_action_preview()

        # extra options
        self.ui.cbx_action_options.addItems(self._prune_options)
        self.ui.cbx_action_options.setCurrentText('Left')
        self.ui.cbx_action_options.setSizeAdjustPolicy(QComboBox.AdjustToContents)

        self.ui.txt_action_edt.setText('Distance:')
        validator = QxDoubleValidator()
        validator.setBottom(0.0)
        self.ui.edt_action.setValidator(validator)
        self.ui.edt_action.setText(str(self._merge_tol))

        # advanced options = for now just for prune
        self.ui.txt_action_options_2.setText('Join method:')
        self.ui.cbx_action_options_2.addItems(self._prune_options_2)
        self.ui.cbx_action_options_2.setCurrentText('mitre')

        self.ui.chk_action_2.setText('Set mitre limit')
        self.ui.chk_action_2.setChecked(True)

        validator_2 = QxDoubleValidator()
        validator_2.setBottom(0.0)
        self.ui.edt_action_2.setValidator(validator)
        self.ui.edt_action_2.setText(str(self._mitre_limit))

        self.ui.txt_action_options_3.setText('Debug arcs:')
        self.ui.cbx_action_options_3.addItems(self._prune_options_3)
        self.ui.cbx_action_options_3.setCurrentText('none')

        self.ui.txt_action_edt_3.setText('Num steps:')
        self.ui.edt_action_3.setText(str(self._num_steps))

        self.ui.chk_advanced.setChecked(False)
        self._on_chk_advanced_options()

        p = self.ui.txt_msg_result.palette()
        color = QColor('red')
        p.setColor(QPalette.WindowText, color)
        self.ui.txt_msg_result.setPalette(p)
        self.ui.txt_msg_result.setHidden(True)
        self.ui.txt_msg_info.setHidden(True)

        self._connect_slots()

        self._allow_display_update = True
        self._in_setup = False

    """
    Data model/view setup
    """
    def _setup_action_preview(self):
        """Sets up the action preview."""
        self._action_preview = ParametersWebPage(self)
        self.ui.vlay_plot.addWidget(self._action_preview)

    def _setup_model(self, is_primary):
        """Sets up the model."""
        if is_primary:
            self.p_model = QxPandasTableModelCheapHeaders(self._ss_df, self, True, self._geom_type)
            self.p_model.set_read_only_columns({i for i in range(1, self.p_model.columnCount())})
            self.p_model.set_show_nan_as_blank(True)
            self.p_model.finished_sort.connect(self._reset_cur_sel)
            self.p_model.layoutChanged.connect(self._update_primary_ss_title)
        else:
            self.s_model = QxPandasTableModelCheapHeaders(self._secondary_ss_df, self, False, self._geom_type)
            self.s_model.set_read_only_columns({i for i in range(self.s_model.columnCount())})
            self.s_model.set_show_nan_as_blank(True)
            self.s_model.finished_sort.connect(self._reset_cur_sel)
            self.s_model.layoutChanged.connect(self._update_secondary_ss_title)

    def _setup_table(self, the_table, is_primary):
        """Sets up the table.

        Args:
            the_table (QTableView): The table we are setting up
            is_primary (bool): True if primary table
        """
        the_table.size_to_contents = True
        widget_builder.style_table_view(the_table)

        # Resize columns and set to interactive resize mode
        horizontal_header = the_table.horizontalHeader()
        horizontal_header.setSectionResizeMode(QHeaderView.Interactive)
        the_table.resizeColumnsToContents()
        horizontal_header.setStretchLastSection(True)

        # We don't want the vertical header
        vertical_header = the_table.verticalHeader()
        vertical_header.hide()

        # Set selection behavior to single rows
        the_table.setSelectionMode(QAbstractItemView.SingleSelection)
        the_table.setSelectionBehavior(QAbstractItemView.SelectRows)

        p = the_table.palette()
        if not is_primary:
            p.setColor(QPalette.Highlight, QColor('green'))
        p.setBrush(p.Inactive, p.Highlight, p.brush(p.Highlight))
        the_table.setPalette(p)

    """
    Widget setup
    """
    def _connect_slots(self):
        """Connect Qt widget signal/slots."""
        # auto updates
        self.ui.chk_auto_update.clicked.connect(self._on_chk_auto_update)
        self.ui.btn_create_preview.clicked.connect(self._on_create_preview)

        # spreadsheets
        self.ui.tbl_primary.selectionModel().selectionChanged.connect(self._on_change_sel_primary)
        self.ui.tbl_secondary.selectionModel().selectionChanged.connect(self._on_change_sel_secondary)
        self.ui.btn_clear_sel.clicked.connect(self._on_clear_secondary_sel)

        # display combos
        self._display_delegate.display_type_changed.connect(self._on_change_item_display)

        # filter method
        self.ui.rbt_distance.clicked.connect(self._on_change_filter)
        self.ui.rbt_multiplier.clicked.connect(self._on_change_filter)
        self.ui.spn_multiplier.valueChanged.connect(self._on_change_filter)
        self.ui.edt_dist.editingFinished.connect(self._on_change_filter)

        # zoom to objects
        self.ui.btn_zoom_primary.clicked.connect(self._on_zoom_to_primary)
        self.ui.btn_zoom_secondary.clicked.connect(self._on_zoom_to_secondary)

        # button to switch primary selection
        self.ui.btn_set_as_primary.clicked.connect(self._on_set_as_primary)

        # delete primary item
        self.ui.btn_delete_primary.clicked.connect(self._on_delete_primary)

        # delete secondary item
        self.ui.btn_delete_secondary.clicked.connect(self._on_delete_secondary)

        # basic action controls
        self.ui.cbx_action.currentIndexChanged.connect(self._on_change_action)
        self.ui.btn_accept_action.clicked.connect(self._on_accept_action)
        self.ui.btn_cancel.clicked.connect(self.reject)
        self.ui.btn_accept_all.clicked.connect(self.accept)

        # extra action options
        self.ui.cbx_action_options.currentIndexChanged.connect(self._on_change_action_option)
        self.ui.edt_action.editingFinished.connect(self._on_change_action_parameter)

        # extra advanced action options
        self.ui.chk_advanced.clicked.connect(self._on_chk_advanced_options)
        self.ui.cbx_action_options_2.currentIndexChanged.connect(self._action_options_2_changed)
        self.ui.chk_action_2.clicked.connect(self._checked_action_2)
        self.ui.edt_action_2.editingFinished.connect(self._on_change_action_parameter)
        self.ui.cbx_action_options_3.currentIndexChanged.connect(self._on_change_action_parameter)
        self.ui.edt_action_3.editingFinished.connect(self._on_change_action_parameter)

        # display map
        self.ui.chk_show_map.clicked.connect(self._on_chk_display_map)

        # fill polys
        self.ui.chk_fill_polys.clicked.connect(self._on_chk_fill_polys)

    """
    Slots
    """
    def _set_geom_type(self):
        """Show/hide rows in the levee results table based on check status."""
        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_GEOM_TYPE

        self._sel_geom_idx = None
        self._geom_type_shapely = 'LineString' if self._geom_type == 'Arc' else 'Polygon'

        self._sel_secondary_geom_idx = None
        self._sel_action = 'Delete'

        self.ui.chk_fill_polys.setHidden(self._geom_type == 'Arc')

        self._change_geom_type_for_table(self.ui.tbl_primary)
        self._change_geom_type_for_table(self.ui.tbl_secondary)

        self.ui.cbx_action.blockSignals(True)
        self.ui.cbx_action.clear()
        if self._geom_type == 'Arc':
            self.ui.cbx_action.addItems(self._arc_actions)
        else:
            self.ui.cbx_action.addItems(self._poly_actions)
        self.ui.cbx_action.blockSignals(False)
        self.ui.cbx_action.setCurrentText(self._sel_action)
        self._update_current_action()

        self._update_map_view = True
        self._map_viewer_params = None
        self._sel_first_selectable_primary_object()

        if self._update_trigger == Triggers.TE_GEOM_TYPE:
            self._zoom_to_smallest_object()
            self._do_updates()

    def _change_geom_type_for_table(self, the_table):
        """Change the geometry type to be displayed.

        Args:
            the_table (QTableView): The table we are setting up
        """
        is_primary = the_table.model()._is_primary
        the_table.model()._set_geom_type(self._geom_type)
        float_delegate = FloatDelegate()
        for col in the_table.model()._float_sort_types:
            the_table.setItemDelegateForColumn(col, float_delegate)
        hide_list = the_table.model()._primary_hide_list if is_primary else the_table.model()._secondary_hide_list
        for hide_col in hide_list:
            the_table.setColumnHidden(hide_col, True)
        the_table.resizeColumnsToContents()
        the_table.horizontalHeader().stretchLastSection()

        if is_primary:
            self._update_primary_ss_title()
        else:
            self._update_secondary_ss_title()

    def _zoom_to_smallest_object(self):
        """Viewers zoom to the smallest object."""
        if self._sel_secondary_geom_idx is None:
            self._on_zoom_to_primary()
            return

        secondary_item = self._geoms[self._sel_secondary_geom_idx]
        if self._sel_geom_idx is not None:
            primary_item = self._sel_geom_item

        if self._geom_type == 'Polygon':
            if primary_item.area <= secondary_item.area:
                self._on_zoom_to_primary()
                return
            self._on_zoom_to_secondary()
            return

        if primary_item.length <= secondary_item.length:
            self._on_zoom_to_primary()
            return

        self._on_zoom_to_secondary()
        return

    def _on_zoom_to_primary(self):
        """Viewers zoom to the primary object."""
        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_ZOOM_TO_PRIMARY

        if self._sel_geom_idx or self._sel_geom_idx_multi:
            if self._sel_action not in self._multi_select_actions:
                zoom_extents = deepcopy(self._latlon_extents[self._sel_geom_idx])
            else:
                zoom_extents = deepcopy(self._latlon_extents[self._sel_geom_idx_multi[0]])
                for idx in self._sel_geom_idx_multi[1:]:
                    sel_bounds = deepcopy(self._latlon_extents[idx])
                    zoom_extents[0][0] = min(sel_bounds[0][0], zoom_extents[0][0])
                    zoom_extents[0][1] = min(sel_bounds[0][1], zoom_extents[0][1])
                    zoom_extents[1][0] = max(sel_bounds[1][0], zoom_extents[1][0])
                    zoom_extents[1][1] = max(sel_bounds[1][1], zoom_extents[1][1])

            if self._multiplier > 0.0:
                dx = (self._multiplier) * (zoom_extents[1][0] - zoom_extents[0][0])
                dy = (self._multiplier) * (zoom_extents[1][1] - zoom_extents[0][1])
                zoom_extents[0][0] -= dx / 2
                zoom_extents[0][1] -= dy / 2
                zoom_extents[1][0] += dx / 2
                zoom_extents[1][1] += dy / 2

            self._action_display_params = None
            self._action_preview_extents = zoom_extents

            self._map_viewer_params = None
            self._map_viewer_extents = zoom_extents
            self._update_map_view = True

        # update the display
        if self._update_trigger == Triggers.TE_ZOOM_TO_PRIMARY:
            self._do_updates(recreate_action_items=False)

    def _on_zoom_to_secondary(self):
        """Viewers zoom to the secondary object."""
        if not self._sel_secondary_geom_idx:
            return

        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_ZOOM_TO_SECONDARY

        zoom_extents = deepcopy(self._latlon_extents[self._sel_secondary_geom_idx])
        if self._multiplier > 0.0:
            dx = (self._multiplier) * (zoom_extents[1][0] - zoom_extents[0][0])
            dy = (self._multiplier) * (zoom_extents[1][1] - zoom_extents[0][1])
            zoom_extents[0][0] -= dx / 2
            zoom_extents[0][1] -= dy / 2
            zoom_extents[1][0] += dx / 2
            zoom_extents[1][1] += dy / 2

        self._action_display_params = None
        self._action_preview_extents = zoom_extents

        self._map_viewer_params = None
        self._map_viewer_extents = zoom_extents
        self._update_map_view = True

        # update the display
        if self._update_trigger == Triggers.TE_ZOOM_TO_SECONDARY:
            self._do_updates(recreate_action_items=False)

    def _on_change_filter(self):
        """Change the multiplier for the bounding box."""
        self._filter_method = FM_BOUNDS_MULT if self.ui.rbt_multiplier.isChecked() else FM_MAX_DIST
        self.ui.spn_multiplier.setEnabled(self._filter_method == FM_BOUNDS_MULT)
        self.ui.edt_dist.setEnabled(self._filter_method == FM_MAX_DIST)

        self._multiplier = float(self.ui.spn_multiplier.text())
        self._max_dist = float(self.ui.edt_dist.text())

        if self._in_setup:
            return

        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_FILTER

        self._update_map_view = True
        self._map_viewer_params = None

        # find intersecting geometries
        self._get_secondary_items_for_display()
        self._populate_secondary_lists()

        # update the display
        if self._update_trigger == Triggers.TE_FILTER:
            self._on_zoom_to_primary()
            self._do_updates()

    def _on_create_preview(self):
        """Change the selected primary object."""
        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_CREATE_PREVIEW

        if self._update_trigger == Triggers.TE_CREATE_PREVIEW:
            self._zoom_to_smallest_object()
            self._do_updates()

    def _on_change_sel_primary(self):
        """Change the selected primary object."""
        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_PRIMARY_SEL

        # clear
        self._addtl_mod_geom_list_idxs.clear()

        # is this selection ok?
        if self.ui.tbl_primary.selectionModel().hasSelection() and not self._is_sel_item_valid(True):
            # message box about why this can't be selected
            if self._update_trigger == Triggers.TE_PRIMARY_SEL and not self._iterating_for_selectable:
                # what is the display type
                geom_idx = self._get_prop_for_selected_row(True, STR_GEOM_IDX)
                disp_type = self._get_prop_for_selected_row(True, STR_DISP_TYPE)

                if disp_type == 'Simplified':
                    opt = message_box.message_with_n_buttons(self, 'Simplified objects cannot be selected for actions.',
                                                             'SMS', ['Switch to Exclude', 'Switch to Normal', 'OK'], 2,
                                                             2)
                    tmp_opts = ['Exclude', 'Normal', 'Simplified']
                else:
                    opt = message_box.message_with_n_buttons(self, 'Excluded objects cannot be selected for actions.',
                                                             'SMS', ['Switch to Simplified', 'Switch to Normal', 'OK'],
                                                             2, 2)
                    tmp_opts = ['Simplified', 'Normal', 'Exclude']

                # update the display type
                disp_type = tmp_opts[opt]
                self._update_prop_for_geom_idx(geom_idx, STR_DISP_TYPE, disp_type)

        # check again to see if it is still not a valid selection
        if self.ui.tbl_primary.selectionModel().hasSelection() and not self._is_sel_item_valid(True):
            cur_sel_row = self.ui.tbl_primary.selectedIndexes()[0].row()
            self._sel_first_selectable_primary_object(cur_sel_row)
        else:
            self._update_primary_selection()

        self._action_display_params = None
        if self._update_trigger == Triggers.TE_PRIMARY_SEL:
            self._zoom_to_smallest_object()
            self._do_updates()

    def _update_primary_selection(self):
        """Update selection variables for the primary selection."""
        self._sel_secondary_geom_idx = None
        self._update_map_view = True
        self._map_viewer_params = None

        # do we have a selection?
        if self.ui.tbl_primary.selectionModel().hasSelection() is False or len(self._ss_df) == 0:
            self._sel_geom_item = None
            self._sel_geom_locs_latlon = None
            self._sel_geom_idx = None

            self._sel_geom_item_multi = None
            self._sel_geom_idx_multi = None
            self._sel_geom_locs_latlon_multi = None

            self._primary_display.clear()
            self._map_viewer.clear()
            self._clear_preview_window()
            self._action_preview_extents.clear()
            return

        # what is the selected geometry?
        if self._sel_action not in self._multi_select_actions:
            self._sel_geom_idx = self._get_prop_for_selected_row(True, STR_GEOM_IDX)
            self._sel_geom_item = self._geoms[self._sel_geom_idx]
            self._sel_geom_locs_latlon = self._geom_latlon_locs[self._sel_geom_idx]

            # save the geometry and the arc id for the tooltip
            self._primary_display = [(self._sel_geom_locs_latlon, self._get_prop_for_selected_row(True, STR_ID))]
        else:
            self._sel_geom_idx_multi = self._get_prop_for_selected_row_multi(STR_GEOM_IDX)
            self._sel_geom_item_multi = [self._geoms[idx] for idx in self._sel_geom_idx_multi]
            self._sel_geom_locs_latlon_multi = [self._geom_latlon_locs[idx] for idx in self._sel_geom_idx_multi]
            feat_ids = self._get_prop_for_selected_row_multi(STR_ID)
            self._primary_display.clear()
            for i, _ in enumerate(self._sel_geom_idx_multi):
                self._primary_display.append((self._sel_geom_locs_latlon_multi[i], feat_ids[i]))

        # find intersecting geometries
        self._get_secondary_items_for_display()
        self._populate_secondary_lists()

    def _get_ss_row_for_geom_idx(self, is_primary, geom_idx):
        """Get the selected item in the spreadsheet and get the dataframe index associated with it.

        Args:
            is_primary (bool): True if the primary spreadsheet
            geom_idx (int): The geometry index
        Returns:
            row (int): Table row that has the geometry index
        """
        df = self._ss_df if is_primary else self._secondary_ss_df
        if len(df.loc[df[STR_GEOM_IDX] == geom_idx]) == 0:
            return 0
        return df.loc[df[STR_GEOM_IDX] == geom_idx].index[0]

    def _get_ss_row_for_feat_id(self, is_primary, feat_idx):
        """Get the selected item in the spreadsheet and get the dataframe index associated with it.

        Args:
            is_primary (bool): True if the primary spreadsheet
            feat_idx (int): The geometry index
        Returns:
            row (int): Table row that has the geometry index
        """
        df = self._ss_df if is_primary else self._secondary_ss_df
        if len(df.loc[df[STR_ID] == feat_idx]) == 0:
            return -1
        return df.loc[df[STR_ID] == feat_idx].index[0]

    def _get_prop_for_geom_idx(self, property, geom_idx):
        """Get the selected item in the spreadsheet and get the dataframe index associated with it.

        Args:
            property (str): Property we want to retrieve
            geom_idx (int): Geometry index
        Returns:
            (object): Desired property
        """
        return self._ss_df.loc[self._ss_df[STR_GEOM_IDX].eq(geom_idx), property].squeeze()

    def _get_prop_for_selected_row_multi(self, property):
        """Get the selected item in the spreadsheet and get the dataframe index associated with it.

        Args:
            property (str): The property we want
        Returns:
            (object): The value
        """
        # what is the selected geometry?
        sel_selection = self.ui.tbl_primary.selectionModel().selection()  # only primary multi-selects
        prop = None
        if len(sel_selection) > 0:
            s_i = sel_selection.indexes()
            rows = {s_i[sel].row() for sel in range(len(s_i))}
            prop = [self._get_prop_for_row(self.ui.tbl_primary, row, property) for row in rows]
        return prop

    def _get_prop_for_selected_row(self, is_primary, property):
        """Get the selected item in the spreadsheet and get the dataframe index associated with it.

        Args:
            is_primary (bool): True if we are using the primary spreadsheet
            property (str): The property we want
        Returns:
            df_idx (int), ss_idx (int): Geometry idx for geom_df, row selection for ss_df
        """
        prop = -1
        # what is the selected geometry?
        the_tbl = self.ui.tbl_primary if is_primary else self.ui.tbl_secondary
        sel_selection = the_tbl.selectionModel().selection()

        if len(sel_selection) != 0:
            row = sel_selection.indexes()[0].row()
            prop = self._get_prop_for_row(the_tbl, row, property)
        return prop

    def _get_prop_for_row(self, tbl, row, property):
        """Get the selected item in the spreadsheet and get the dataframe index associated with it.

        Args:
            tbl (QTableView): The table we want the property from
            row (int): Row we want the property from
            property (str): The property we want
        Returns:
            (object): The value
        """
        # what is the selected geometry?
        if property == STR_GEOM_IDX:
            column = tbl.model()._col_geom_list_idx
        elif property == STR_ID:
            column = tbl.model()._col_feat_id
        elif property == STR_DIST_TO_P:
            column = tbl.model()._col_dist_to_p
        elif property == STR_DISP_TYPE:
            column = tbl.model()._col_disp_type

        index = tbl.model().index(row, column)
        data = tbl.model().data(index)

        if not data:
            return None

        if column in tbl.model()._float_sort_types:
            return float(data)
        elif column in tbl.model()._int_sort_types:
            return int(data)

        return tbl.model().data(index)

    def _get_simplified_locs(self, geom_idx, update=False):
        """Filter out points.

        Args:
            geom_idx (int): Index of geometry we are simplifying
            update (bool): Force computing the simplified points
        """
        if geom_idx not in self._simplified_geom_latlon_locs.keys() or update:
            orig_locs_lists = self._geom_latlon_locs[geom_idx]
            simplified_locs = orig_locs_lists.copy()
            for orig_locs in orig_locs_lists:
                num_skip = 10
                num_segs = len(orig_locs)
                while num_segs / num_skip > self._simplify_tol / 20.0:
                    num_skip *= 2.0

                if len(orig_locs) > num_skip * 2.0:
                    simplified_locs.append([orig for idx, orig in enumerate(orig_locs) if idx % num_skip == 0])

            self._simplified_geom_latlon_locs[geom_idx] = simplified_locs

        return self._simplified_geom_latlon_locs[geom_idx]

    def _get_secondary_items_for_display(self):
        """See if we have a containing polygon."""
        self._secondary_geoms_in_bounds = []
        self._sel_secondary_display.clear()

        # find intersecting geometries
        if self._sel_action not in self._multi_select_actions:
            bounds_for_rtree = self._sel_geom_item.bounds
            non_geo_primary = self._geoms_for_props[self._sel_geom_idx]
        else:
            bounds_for_rtree = self._sel_geom_item_multi[0].bounds
            if len(self._sel_geom_item_multi) > 1:
                min_x = bounds_for_rtree[0]
                min_y = bounds_for_rtree[1]
                max_x = bounds_for_rtree[2]
                max_y = bounds_for_rtree[3]
                for sel_item in self._sel_geom_item_multi[1:]:
                    sel_bounds = sel_item.bounds
                    min_x = min(sel_bounds[0], min_x)
                    min_y = min(sel_bounds[1], min_y)
                    max_x = max(sel_bounds[2], max_x)
                    max_y = max(sel_bounds[3], max_y)
                bounds_for_rtree = (min_x, min_y, max_x, max_y)
                non_geo_primary = None
            else:
                non_geo_primary = self._geoms_for_props[self._sel_geom_idx_multi[0]]

        if self._filter_method == FM_BOUNDS_MULT:
            dx = self._multiplier * abs(bounds_for_rtree[2] - bounds_for_rtree[0]) / 2.0
            dy = self._multiplier * (bounds_for_rtree[3] - bounds_for_rtree[1]) / 2.0
        else:
            if self._is_geo:
                dist = self._convert(self._max_dist, 0.0) * 1.5
            else:
                dist = self._max_dist * 1.5
            dx = dy = dist

        min_x = bounds_for_rtree[0] - dx
        max_x = bounds_for_rtree[2] + dx
        min_y = bounds_for_rtree[1] - dy
        max_y = bounds_for_rtree[3] + dy
        bounds_for_rtree = (min_x, min_y, max_x, max_y)

        candidate_indices = self._rtree.intersection(bounds_for_rtree)

        negative_dist_idxs = []
        if self._geom_type == 'Polygon' and non_geo_primary is not None:
            g_idx = self._sel_geom_idx_multi[0] if self._sel_geom_idx_multi else self._sel_geom_idx
            negative_dist_idxs = self._get_hole_polys(g_idx)
            c_poly = self._get_containing_poly(g_idx)
            if c_poly:
                negative_dist_idxs.append(c_poly)

        for geom_idx in candidate_indices:
            if geom_idx == self._sel_geom_idx or geom_idx in self._deleted_geom_idxs:
                continue
            if self._sel_geom_idx_multi and geom_idx in self._sel_geom_idx_multi:
                continue
            if self._get_prop_for_geom_idx(STR_DISP_TYPE, geom_idx) == 'Exclude':
                continue

            if non_geo_primary is not None:
                non_geo_secondary = self._geoms_for_props[geom_idx]
                if self._geom_type == 'Arc':
                    distance = non_geo_primary.distance(non_geo_secondary)
                else:
                    distance = non_geo_primary.exterior.distance(non_geo_secondary.exterior)
                    if geom_idx in negative_dist_idxs:
                        distance = -distance

                if self._filter_method == FM_BOUNDS_MULT or distance < self._max_dist:
                    self._secondary_geoms_in_bounds.append(geom_idx)
                    self._update_prop_for_geom_idx(geom_idx, STR_DIST_TO_P, distance)

    def _populate_secondary_lists(self):
        """Find which items belong in the secondary list and put them there."""
        self.ui.tbl_secondary.resizeColumnsToContents()
        self.ui.tbl_secondary.horizontalHeader().stretchLastSection()

        self._secondary_display.clear()

        # get a dataframe of just the geometry that is in bounds
        self._secondary_ss_df = pd.DataFrame(
            self._ss_df.loc[self._ss_df[STR_GEOM_IDX].isin(self._secondary_geoms_in_bounds)])
        self._secondary_ss_df.reset_index(drop=True)
        self.s_model.update_data_frame(self._secondary_ss_df)

        for geom_idx_to_show in self._secondary_geoms_in_bounds:
            if self._get_prop_for_geom_idx(STR_DISP_TYPE, geom_idx_to_show) == 'Simplified':
                intersecting_locs = self._get_simplified_locs(geom_idx_to_show)
            else:
                intersecting_locs = self._geom_latlon_locs[geom_idx_to_show]
            self._secondary_display.append((intersecting_locs, self._get_prop_for_geom_idx(STR_ID, geom_idx_to_show)))

        # select the first secondary item
        if self._sel_action in self._actions_that_use_secondary:
            self._sel_first_selectable_secondary_object()
        else:
            self.ui.tbl_secondary.selectionModel().clearSelection()

        self.ui.btn_set_as_primary.setEnabled(self._sel_action in self._actions_that_use_secondary)
        self.ui.btn_zoom_secondary.setEnabled(self._sel_action in self._actions_that_use_secondary)
        self.ui.btn_delete_secondary.setEnabled(self._sel_action in self._actions_that_use_secondary)

    def _on_change_sel_secondary(self):
        """Handle changing the selection in the secondary spreadsheet."""
        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_SECONDARY_SEL

        # clear
        self._addtl_mod_geom_list_idxs.clear()

        # is this selection ok?
        has_secondary_sel = self.ui.tbl_secondary.selectionModel().hasSelection()
        if has_secondary_sel and not self._is_sel_item_valid(False, self._iterating_for_selectable):
            # message box about why this can't be selected
            if self._update_trigger == Triggers.TE_SECONDARY_SEL and not self._iterating_for_selectable:
                # only show the message if they actually clicked on the object
                message_box.message_with_ok(self, 'Simplified objects cannot be selected for actions.', 'SMS')
            cur_sel_row = self.ui.tbl_secondary.selectedIndexes()[0].row()
            self._sel_first_selectable_secondary_object(cur_sel_row)
        else:
            self._update_secondary_selection()

        self._update_map_view = True

        if self._update_trigger == Triggers.TE_SECONDARY_SEL:
            self._zoom_to_smallest_object()
            self._do_updates()

    def _update_secondary_selection(self):
        """Update all selection variables."""
        self._msg_info = None
        if self.ui.tbl_secondary.selectionModel().hasSelection() is False:
            self._sel_secondary_display.clear()
            self._sel_secondary_geom_idx = None
            self._sel_secondary_item_extents = None
            self.ui.btn_set_as_primary.setEnabled(False)
            self.ui.btn_zoom_secondary.setEnabled(False)
            self.ui.btn_delete_secondary.setEnabled(False)
        else:
            # what is the selected geometry?
            self._sel_secondary_geom_idx = self._get_prop_for_selected_row(False, STR_GEOM_IDX)
            sel_secondary_item_latlon_locs = self._geom_latlon_locs[self._sel_secondary_geom_idx]
            self._sel_secondary_item_extents = deepcopy(self._latlon_extents[self._sel_secondary_geom_idx])

            # save the geometry and the id for the tooltip
            self._sel_secondary_display = [(sel_secondary_item_latlon_locs,
                                            self._get_prop_for_selected_row(False, STR_ID))]
            self.ui.btn_set_as_primary.setEnabled(True)
            self.ui.btn_zoom_secondary.setEnabled(True)
            if self._sel_action in self._actions_that_use_secondary:
                self.ui.btn_delete_secondary.setEnabled(True)

        self._update_map_view = True

    def _on_change_action_option(self):
        """Change an option for the action.

        Args:
            geoms_to_draw (list): List of geometry to draw - first is primary
            extents (list): List of extents
        """
        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_ACTION_OPT
            self._do_updates()

    def _on_change_item_display(self, index):
        """Change the display status of an item.

        Args:
            index (object): index of item changed
        """
        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_ITEM_DISPLAY

        display_type = self._display_types[index]
        self._update_prop_for_geom_idx(self._sel_geom_idx, STR_DISP_TYPE, display_type)

        # if this is simplified or excluded, it cannot be selected in the primary spreadsheet (because we can't
        # perform actions on it)
        if display_type != 'Normal':
            table_indexes = self.ui.tbl_primary.selectedIndexes()
            current_row = table_indexes[0].row()
            # get the next selectable item
            self._sel_first_selectable_primary_object(current_row)

        # check on what is displayed in the secondary list - excluded items do not need to be listed at all and
        # simplified items should not be selectable
        if self._update_trigger == Triggers.TE_ITEM_DISPLAY:
            self._do_updates()

    def _is_sel_item_valid(self, is_primary, check_for_neg_val=False):
        """Only allow the item to be selected if the display is "normal"."""
        if self._get_prop_for_selected_row(is_primary, STR_DISP_TYPE) != 'Normal':
            return False
        if check_for_neg_val and not is_primary and self._get_prop_for_selected_row(is_primary, STR_DIST_TO_P) < 0.0:
            # don't want to select one with a negative distance
            return False

        return True

    def _sel_first_primary_object(self):
        """Use this for when we can't have arguments."""
        if self._newest_geom_idx is not None:
            self._cur_sel_row_primary = self._get_ss_row_for_geom_idx(True, self._newest_geom_idx)
        self._sel_first_selectable_primary_object(self._cur_sel_row_primary)
        if self._in_setup is False and self._update_trigger == Triggers.TE_NONE:
            self._zoom_to_smallest_object()
            self._do_updates()

    def _scroll_to_selection(self, is_primary):
        """Selects an object in the primary spreadsheet that is "Normal"."""
        the_tbl = self.ui.tbl_primary if is_primary else self.ui.tbl_secondary
        selected = the_tbl.selectionModel().selectedIndexes()
        if selected:
            sel_single = selected[0]
            the_tbl.scrollTo(sel_single, QAbstractItemView.PositionAtCenter)
            the_tbl.update()

    def _sel_first_selectable_primary_object(self, starting_index=0):
        """Selects an object in the primary spreadsheet that is "Normal"."""
        if self._in_setup:
            return

        row_count = self.p_model.rowCount()
        self.ui.tbl_primary.selectionModel().blockSignals(True)
        self._iterating_for_selectable = True
        self.ui.tbl_primary.selectRow(starting_index)
        valid_selection = self._is_sel_item_valid(True)
        index = starting_index + 1 if starting_index < row_count else 0
        while valid_selection is False and index != starting_index:
            self.ui.tbl_primary.selectRow(index)
            valid_selection = self._is_sel_item_valid(True)
            if valid_selection is False:
                index = index + 1 if index < row_count else 0
        self._iterating_for_selectable = False
        self.ui.tbl_primary.selectionModel().blockSignals(False)

        self._scroll_to_selection(True)

        if index == starting_index:
            self.ui.tbl_primary.selectionModel().clearSelection()

        self._update_primary_selection()

    def _sel_first_selectable_secondary_object(self, starting_index=0, check_neg_val=True):
        """Selects an object in the secondary spreadsheet."""
        if self._in_setup or len(self._secondary_display) == 0:
            return

        row_count = self.s_model.rowCount()
        self._iterating_for_selectable = True
        self.ui.tbl_secondary.selectionModel().blockSignals(True)
        self.ui.tbl_secondary.selectRow(starting_index)
        valid_selection = self._is_sel_item_valid(False, check_neg_val)
        index = starting_index + 1 if starting_index < row_count - 1 else 0
        while valid_selection is False and index != starting_index:
            self.ui.tbl_secondary.selectRow(index)
            valid_selection = self._is_sel_item_valid(False, True)
            if valid_selection is False:
                index = index + 1 if index < row_count else 0

        self._iterating_for_selectable = False
        self.ui.tbl_secondary.selectionModel().blockSignals(False)
        if valid_selection is False:
            if check_neg_val:
                return self._sel_first_selectable_secondary_object(starting_index, False)
            else:
                self.ui.tbl_secondary.selectionModel().clearSelection()
        self._update_secondary_selection()

        self._scroll_to_selection(False)

    def _create_action_geoms(self):
        """Perform the action and create the new geometries."""
        self._msg_action = None
        if len(self._primary_display) == 0:
            self._clear_preview_window()
            return

        self._reset_display_colors()
        self._display_dashed_item = False

        self._action_display = self._primary_display.copy()
        self._mod_latlon_extents.clear()
        if self._sel_action not in self._multi_select_actions:
            self._mod_latlon_extents.append(deepcopy(self._latlon_extents[self._sel_geom_idx]))
        else:
            self._mod_latlon_extents = [deepcopy(self._latlon_extents[idx]) for idx in self._sel_geom_idx_multi]
        self._draw_sel_secondary_for_action_preview = True
        self._addtl_mod_geom_list_idxs.clear()
        debug_arcs = []
        self._debug_arcs_display = []
        if self._sel_action == 'Prune':
            self._display_dashed_item = True
            self._mod_geoms, debug_arcs = self._do_prune_arc()
        elif self._sel_action == 'Snap':
            self._mod_geoms = self._do_snap_arc()
        elif self._sel_action == 'Collapse':
            self._mod_geoms = self._do_collapse_arc(self._sel_geom_item,
                                                    self.ui.cbx_action_options.currentText() == 'Start')
            self._display_colors[0] = 'gray'  # we want the modified arcs to just be gray, not blue
        elif self._sel_action == 'Concatenate':
            self._mod_geoms = self._do_concatenate_arcs()
        elif self._sel_action == 'Curvature Redistribution':
            self._max_delta = float(self.ui.edt_action.text())
            self._min_seg_length = float(self.ui.edt_action_4.text())
            self._mod_geoms = self._do_curvature_redist()
            self._display_dashed_item = True
        elif self._sel_action == 'Merge':
            self._mod_geoms = self._do_merge_polygons()
        elif self._sel_action == 'Delete':
            self._display_dashed_item = True

        # unless we are deleting, we need to transform the modified object to latlon
        action = self._sel_action
        if action == 'Delete' or (self._mod_geoms is not None and len(self._mod_geoms) == 0 and action == 'Collapse'):
            # self._action_display.remove(self._action_display[0])
            self._action_display.clear()
        elif self._mod_geoms is not None:
            self._action_display.clear()
            self._mod_latlon_locs.clear()
            self._mod_latlon_extents.clear()
            no_draw_secondary = ['Merge', 'Concatenate']
            self._draw_sel_secondary_for_action_preview = self._sel_action not in no_draw_secondary
            for geom in self._mod_geoms:
                latlon_locs, extents = self._transform_geom(geom)
                self._mod_latlon_locs.append(latlon_locs)
                self._mod_latlon_extents.append(extents)
                self._action_display.append((latlon_locs, -1))
            if self._draw_sel_secondary_for_action_preview and self._sel_secondary_geom_idx:
                self._mod_latlon_extents.append(deepcopy(self._latlon_extents[self._sel_secondary_geom_idx]))
        else:
            if not self._msg_action:
                self._msg_action = 'Action result: No impact.'
            if self._sel_secondary_geom_idx is not None:
                self._mod_latlon_extents.append(deepcopy(self._latlon_extents[self._sel_secondary_geom_idx]))

        if self._geom_type == 'Arc':
            for debug_arc in debug_arcs:
                if self._is_geo:
                    debug_latlon_locs = [[(pt[0], pt[1]) for pt in gsh.pts_from_ls_tuple(debug_arc)]]
                else:
                    debug_latlon_locs, _ = self._transform_geom(debug_arc)
                self._debug_arcs_display.append((debug_latlon_locs, -1))

    def _do_merge_polygons(self):
        """Create the new geometries from the merged polygons."""
        self._removed_inner_polys.clear()
        if self._sel_secondary_geom_idx is None:
            self._msg_action = 'A secondary polygon must be selected for this action.'
            return None

        # is one of these a hole in the other?
        self._merge_is_difference = False
        primary_holes = self._get_hole_polys(self._sel_geom_idx)
        swap_polys = False
        if self._sel_secondary_geom_idx in primary_holes:
            self._merge_is_difference = True
        else:
            secondary_holes = self._get_hole_polys(self._sel_secondary_geom_idx)
            if self._sel_geom_idx in secondary_holes:
                self._merge_is_difference = True
                swap_polys = True

        self._merge_is_union = not self._merge_is_difference

        poly_1 = self._sel_geom_item
        poly_2 = self._geoms[self._sel_secondary_geom_idx]

        if self._merge_is_union:
            merged = unary_union([poly_1, poly_2])
            if merged and merged.geom_type == 'Polygon' and merged.is_valid:
                return [merged]

        err_msg = ('Action result: No impact. May be caused by not enough adjacent points within the '
                   'specified tolerance. Try increasing the tolerance.')

        # if we are here, we are doing something with tolerance
        self._merge_tol = float(self.ui.edt_action.text())

        # we want non-geographic polys if we are using a specified distance since they don't match up
        non_geo_poly_1 = self._geoms_for_props[self._sel_geom_idx]
        non_geo_poly_2 = self._geoms_for_props[self._sel_secondary_geom_idx]

        if self._merge_is_union:
            merger = IntersectPolygonsByTolerance(self._is_geo, self._merge_tol)
            merged = merger.do_intersect(poly_1, poly_2, non_geo_poly_1, non_geo_poly_2, True)
            self._msg_action = None if merged else err_msg
            return [merged] if merged else None

        # if we are here, it is difference intersection
        # first make sure that one polygon is contained in the other
        poly_1_idx = self._sel_geom_idx
        poly_2_idx = self._sel_secondary_geom_idx
        if swap_polys:
            # swap so that poly_1 is the bigger poly
            poly_1, poly_2 = poly_2, poly_1
            non_geo_poly_1, non_geo_poly_2 = non_geo_poly_2, non_geo_poly_1
            poly_1_idx, poly_2_idx = poly_2_idx, poly_1_idx

        closest_dist = None
        orig_non_geo_small_poly = deepcopy(non_geo_poly_2)

        # before doing the difference intersect with the polygon perimeter, we want to do the holes, which
        # is a union merge
        self._removed_inner_polys.clear()
        inner_polys_idxs = self._get_hole_polys(poly_1_idx)
        # we could have tons of holes to sort through, so make an rtree
        all_hole_boxes = [self._geoms_for_props[hole_idx].bounds for hole_idx in inner_polys_idxs]

        def generator_func():
            for j, b in enumerate(all_hole_boxes):
                yield j, b, b
        tmp_rtree = index.Index(generator_func())

        # update the bounds to include the merge tolerance so we can easily find all hole polys within tol
        min_x = non_geo_poly_2.bounds[0] - self._merge_tol
        max_x = non_geo_poly_2.bounds[2] + self._merge_tol
        min_y = non_geo_poly_2.bounds[1] - self._merge_tol
        max_y = non_geo_poly_2.bounds[3] + self._merge_tol
        bounds_for_rtree = [min_x, min_y, max_x, max_y]
        candidate_idxs = tmp_rtree.intersection(bounds_for_rtree)

        mergeable = dict()
        geom_idxs = [inner_polys_idxs[candidate_idx] for candidate_idx in candidate_idxs]
        for hole_idx in geom_idxs:
            if hole_idx in self._removed_inner_polys or hole_idx == poly_2_idx:
                continue

            hole_geom = self._geoms[hole_idx]
            non_geo_hole = self._geoms_for_props[hole_idx]

            force_merge = non_geo_poly_2.contains(non_geo_hole)

            # find the distance
            if not force_merge:
                hole_dist = non_geo_hole.distance(orig_non_geo_small_poly)
                if not closest_dist:
                    closest_dist = hole_dist
                else:
                    closest_dist = min(closest_dist, hole_dist)

            # is this close enough to merge?
            if force_merge or hole_dist < self._merge_tol:
                if hole_dist in mergeable.keys():
                    mergeable[hole_dist].append((hole_idx, hole_geom, non_geo_hole))
                else:
                    mergeable[hole_dist] = [(hole_idx, hole_geom, non_geo_hole)]

        changed = True
        while changed and len(mergeable) > 0:
            tmp_mergeable = mergeable.copy()
            changed = False

            dists = sorted(list(tmp_mergeable.keys()))
            for dist in dists:
                holes_at_dist = tmp_mergeable[dist].copy()
                for (hole_idx, hole_geom, non_geo_hole) in holes_at_dist:
                    merged_hole = None
                    if dist == 0.0:
                        tmp_hole = unary_union([hole_geom, poly_2])
                        if tmp_hole and tmp_hole.geom_type == 'Polygon' and tmp_hole.is_valid:
                            merged_hole = tmp_hole

                    if merged_hole is None:
                        # maybe the original polygon wasn't touching this hole, but now that it is merged, it might be
                        if non_geo_poly_2.distance(non_geo_hole) == 0.0:
                            tmp_hole = unary_union([hole_geom, poly_2])
                            if tmp_hole and tmp_hole.geom_type == 'Polygon' and tmp_hole.is_valid:
                                merged_hole = tmp_hole
                        if merged_hole is None:
                            merger = IntersectPolygonsByTolerance(self._is_geo, self._merge_tol)
                            merged_hole = merger.do_intersect(hole_geom, poly_2, non_geo_hole, non_geo_poly_2, True)

                    if merged_hole and merged_hole.is_valid:
                        # this merge may put us in tolerance with a polygon that we weren't in tolerance with
                        # before, so we will try again
                        changed = True
                        tmp_mergeable[dist].remove((hole_idx, hole_geom, non_geo_hole))
                        if len(tmp_mergeable[dist]) == 0:
                            del tmp_mergeable[dist]

                        poly_2 = merged_hole
                        non_geo_poly_2 = self._convert_latlon_geom_to_utm(merged_hole) if self._is_geo else merged_hole
                        self._removed_inner_polys.append(hole_idx)

            mergeable = tmp_mergeable

        perims_dist = non_geo_poly_1.exterior.distance(orig_non_geo_small_poly.exterior)
        if not closest_dist:
            closest_dist = perims_dist
        else:
            closest_dist = min(closest_dist, perims_dist)
        self._msg_info = f'Distance between boundaries: {closest_dist:.2f}'
        merger = IntersectPolygonsByTolerance(self._is_geo, self._merge_tol)
        merged = merger.do_intersect(poly_1, poly_2, non_geo_poly_1, non_geo_poly_2, False)

        if (merged is None or not merged.is_valid) and len(self._removed_inner_polys) > 0:
            # no merging with the perimeter, but there was merging with holes
            # don't draw the holes that were merged
            perimeter = gsh.perim_pts_from_sh_poly(poly_1)
            holes = [gsh.perim_pts_from_sh_poly(poly_2)]
            orig_holes = self._get_hole_polys(poly_1_idx)
            for hole_poly_idx in orig_holes:
                if hole_poly_idx in self._removed_inner_polys or hole_poly_idx == poly_2_idx:
                    continue
                holes.append(gsh.perim_pts_from_sh_poly(self._geoms[hole_poly_idx]))

            self._removed_inner_polys.append(poly_2_idx)
            self._addtl_mod_geom_list_idxs.extend(list(self._removed_inner_polys))

            return [shPolygon(perimeter, holes), poly_2]
        elif merged and merged.is_valid:
            # poly_2 was a hole and it should be removed since it merged with the perimeter
            self._removed_inner_polys.append(poly_2_idx)

            holes_to_keep = [idx for idx in inner_polys_idxs if idx not in self._removed_inner_polys]
            perim = gsh.perim_pts_from_sh_poly(merged)
            hole_locs = [gsh.perim_pts_from_sh_poly(self._geoms[hole]) for hole in holes_to_keep]

            # don't draw the holes that were merged
            self._addtl_mod_geom_list_idxs.extend(list(self._removed_inner_polys))
            return [shPolygon(perim, hole_locs)]

        # nothing happened if we got here
        self._msg_action = err_msg
        return None

    def _reset_cur_sel(self, is_primary):
        """Disable/enable items in the advanced options.

        Args:
            sel_geom_idx (int): Selected geom idx before sorting
            is_primary (bool): True if primary spreadsheet
        """
        the_tbl = self.ui.tbl_primary if is_primary else self.ui.tbl_secondary
        if is_primary:
            geom_idx = None
            if self._sel_geom_idx is not None:
                geom_idx = self._sel_geom_idx
            elif self._sel_geom_idx_multi and len(self._sel_geom_idx_multi) > 0:
                geom_idx = self._sel_geom_idx_multi[0]
                self._sel_geom_idx_multi = [geom_idx]
        else:
            geom_idx = self._sel_secondary_geom_idx

        if geom_idx is not None and geom_idx not in self._deleted_geom_idxs:
            row = self._get_ss_row_for_geom_idx(is_primary, geom_idx)
            the_tbl.blockSignals(True)
            the_tbl.selectRow(row)
            the_tbl.blockSignals(False)

    def _on_chk_auto_update(self):
        """Disable/enable items in the advanced options.

        Args:
            geoms_to_draw (list): List of geometry to draw - first is primary
            extents (list): List of extents
        """
        self._do_auto_updates = self.ui.chk_auto_update.isChecked()
        self.ui.btn_create_preview.setEnabled(not self._do_auto_updates)

        if self._do_auto_updates is True:
            self._zoom_to_smallest_object()
            self._do_updates()

    def _on_chk_advanced_options(self):
        """Disable/enable items in the advanced options.

        Args:
            geoms_to_draw (list): List of geometry to draw - first is primary
            extents (list): List of extents
        """
        on = self.ui.chk_advanced.isChecked()
        self.ui.grp_advanced.setVisible(on)
        self._on_change_action_option()

    def _action_options_2_changed(self):
        """Disable/enable items as necessary.

        Args:
            geoms_to_draw (list): List of geometry to draw - first is primary
            extents (list): List of extents
        """
        show = self.ui.cbx_action_options_2.isVisible() and self.ui.cbx_action_options_2.currentText() == 'mitre'
        self.ui.chk_action_2.setVisible(show)
        self._checked_action_2()

    def _checked_action_2(self):
        """Disable/enable items as necessary."""
        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_CHK_ACTION_2

        show = self.ui.chk_action_2.isVisible() and self.ui.chk_action_2.isChecked()
        self.ui.edt_action_2.setVisible(show)

        if self._update_trigger == Triggers.TE_CHK_ACTION_2:
            self._do_updates()

    def _show_hide_messages(self):
        """Show/hide the messages group box."""
        if self._msg_action:
            self.ui.txt_msg_result.setText(self._msg_action)
        if self._msg_info:
            self.ui.txt_msg_info.setText(self._msg_info)

        self.ui.txt_msg_result.setHidden(not self._msg_action)
        self.ui.txt_msg_info.setHidden(not self._msg_info)
        self.ui.grp_messages.setHidden(self.ui.txt_msg_result.isHidden() and self.ui.txt_msg_info.isHidden())

        self.ui.grp_messages.repaint()
        self.ui.txt_msg_result.repaint()
        self.ui.txt_msg_info.repaint()

    def _on_clear_secondary_sel(self):
        """Clear the selection in the secondary spreadsheet."""
        # clear the selection
        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_CLEAR_SECONDARY

        self._sel_secondary_geom_idx = None
        self._sel_secondary_display.clear()
        self._mod_geoms = None
        self.ui.tbl_secondary.selectionModel().clear()
        self._update_secondary_selection()

        if self._update_trigger == Triggers.TE_CLEAR_SECONDARY:
            self._do_updates()

    def _on_change_action(self):
        """Change the action type."""
        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_ACTION_SEL

        self._update_current_action()

        if self._update_trigger == Triggers.TE_ACTION_SEL:
            # self._on_zoom_to_primary()
            self._do_updates()

    def _update_current_action(self):
        """Update controls for the new action selection."""
        prev_action = self._sel_action
        self._sel_action = self.ui.cbx_action.currentText()

        enable_secondary = self._sel_action in self._actions_that_use_secondary
        self.ui.grp_secondary.setEnabled(enable_secondary)
        if not enable_secondary:
            self._on_clear_secondary_sel()

        self.ui.cbx_action_options.blockSignals(True)

        if self._sel_action == 'Delete' or self._sel_action == 'Concatenate':
            self.ui.grp_options.setHidden(True)
            self.ui.chk_advanced.setHidden(True)
            self.ui.grp_advanced.setHidden(True)
            self.ui.txt_action_4.setHidden(True)
            self.ui.edt_action_4.setHidden(True)
        elif self._sel_action == 'Prune':
            self.ui.grp_options.setHidden(False)
            self.ui.chk_advanced.setHidden(False)
            self.ui.cbx_action_options.clear()
            self.ui.cbx_action_options.addItems(self._prune_options)
            self.ui.cbx_action_options.setCurrentText(self._prune_side)
            self.ui.txt_action_options.setText('Side:')
            self.ui.txt_action_edt.setHidden(False)
            self.ui.txt_action_edt.setText('Distance:')
            self.ui.edt_action.setHidden(False)
            self.ui.edt_action.setText(str(self._prune_distance))
            self.ui.cbx_action_options.setHidden(False)
            self._on_chk_advanced_options()
            self.ui.txt_action_4.setHidden(True)
            self.ui.edt_action_4.setHidden(True)
        elif self._sel_action == 'Merge':
            self.ui.grp_options.setHidden(False)
            self.ui.chk_advanced.setHidden(True)
            self.ui.grp_advanced.setHidden(True)
            self.ui.cbx_action_options.setHidden(True)
            self.ui.txt_action_edt.setHidden(False)
            self.ui.edt_action.setHidden(False)
            self.ui.txt_action_options.setHidden(True)
            self.ui.txt_action_4.setHidden(True)
            self.ui.edt_action_4.setHidden(True)
            self.ui.txt_action_edt.setHidden(False)
        elif self._sel_action == 'Snap':
            self.ui.grp_options.setHidden(False)
            self.ui.chk_advanced.setHidden(True)
            self.ui.grp_advanced.setHidden(True)
            self.ui.cbx_action_options.setHidden(True)
            self.ui.txt_action_options.setHidden(True)
            self.ui.txt_action_edt.setHidden(False)
            self.ui.txt_action_edt.setText('Tolerance:')
            self.ui.edt_action.setHidden(False)
            self.ui.edt_action.setText(str(self._snap_tol))
            self.ui.txt_action_4.setHidden(True)
            self.ui.edt_action_4.setHidden(True)
        elif self._sel_action == 'Collapse':
            self.ui.grp_options.setHidden(False)
            self.ui.chk_advanced.setHidden(True)
            self.ui.grp_advanced.setHidden(True)
            self.ui.cbx_action_options.setHidden(False)
            self.ui.cbx_action_options.clear()
            self.ui.cbx_action_options.addItems(self._collapse_options)
            self.ui.txt_action_options.setHidden(False)
            self.ui.txt_action_options.setText('Node to preserve:')
            self.ui.txt_action_edt.setHidden(True)
            self.ui.edt_action.setHidden(True)
            self.ui.txt_action_4.setHidden(True)
            self.ui.edt_action_4.setHidden(True)
        elif self._sel_action == 'Curvature Redistribution':
            self.ui.grp_options.setHidden(False)
            self.ui.chk_advanced.setHidden(True)
            self.ui.grp_advanced.setHidden(True)
            self.ui.cbx_action_options.setHidden(True)
            self.ui.txt_action_options.setHidden(True)
            self.ui.txt_action_edt.setHidden(False)
            self.ui.txt_action_edt.setText('Max delta:')
            self.ui.edt_action.setHidden(False)
            self.ui.edt_action.setText(str(self._max_delta))
            self.ui.txt_action_4.setText('Min segment length:')
            self.ui.txt_action_4.setHidden(False)
            self.ui.edt_action_4.setHidden(False)
            self.ui.edt_action_4.setText(str(self._min_seg_length))

        self.ui.cbx_action_options.blockSignals(False)

        # allow "Delete Primary" if delete is not the action
        self.ui.btn_delete_primary.setHidden(self._sel_action == 'Delete')

        # allow "Delete Secondary" if merge is selected
        self.ui.btn_delete_secondary.setHidden(self._sel_action not in self._actions_that_use_secondary)

        # allow multi-selecting if the action is "Delete"
        if self._sel_action in self._multi_select_actions:
            sm = QAbstractItemView.ExtendedSelection
        else:
            sm = QAbstractItemView.SingleSelection
        self.ui.tbl_primary.setSelectionMode(sm)

        prev_was_multi = prev_action in self._multi_select_actions
        changed_multi_select = (self._sel_action in self._multi_select_actions) != prev_was_multi
        if changed_multi_select and self.ui.tbl_primary.selectionModel().hasSelection():
            self._change_selections_to_single() if prev_was_multi else self._change_selections_to_multi()
        elif self.ui.tbl_primary.selectionModel().hasSelection() is False:
            self._sel_first_selectable_primary_object()

        # this may have resized things (if extra options show up), so make sure we can still see the selected
        # geometry in the spreadsheet
        self._scroll_to_selection(True)

        if enable_secondary and self.ui.tbl_secondary.selectionModel().hasSelection() is False:
            # make sure that a secondary object is selected
            self._sel_first_selectable_secondary_object()

    def _update_secondary_ss_title(self):
        """Update group title."""
        if self._geom_type == 'Arc':
            text = f'Secondary ({self.s_model.rowCount()} Arcs)'
        else:
            text = f'Secondary ({self.s_model.rowCount()} Polygons)'
        self.ui.grp_secondary.setTitle(text)

    def _update_primary_ss_title(self):
        """Update group title."""
        if self._geom_type == 'Arc':
            text = f'Primary ({self.p_model.rowCount()} Arcs)'
        else:
            text = f'Primary ({self.p_model.rowCount()} Polygons)'
        self.ui.grp_primary.setTitle(text)

    def _change_selections_to_multi(self):
        """Set selection selection to multi."""
        self._sel_geom_item_multi = [self._sel_geom_item]
        self._sel_geom_idx_multi = [self._sel_geom_idx]
        self._sel_geom_locs_latlon_multi = [self._sel_geom_locs_latlon]

        self._sel_geom_item = None
        self._sel_geom_locs_latlon = None
        self._sel_geom_idx = None

    def _change_selections_to_single(self):
        """Make multi selections a single selection."""
        self._sel_geom_idx = self._sel_geom_idx_multi[0]
        self._sel_geom_item = self._sel_geom_item_multi[0]
        self._sel_geom_locs_latlon = self._sel_geom_locs_latlon_multi[0]

        self._sel_geom_item_multi = None
        self._sel_geom_idx_multi = None
        self._sel_geom_locs_latlon_multi = None

    def _on_set_as_primary(self):
        """Set the selected secondary geometry to be the selected primary."""
        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_SET_AS_PRIMARY

        # find the row to select as the primary
        p_row = self._get_ss_row_for_geom_idx(True, self._sel_secondary_geom_idx)
        self.ui.tbl_primary.selectRow(p_row)

        if self._update_trigger == Triggers.TE_SET_AS_PRIMARY:
            self._zoom_to_smallest_object()
            self._do_updates()

    def _on_delete_primary(self):
        """Set the selected secondary geometry to be the selected primary."""
        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_DELETE_PRIMARY

        self._delete_geom(self._sel_geom_idx, True, True)

        # clear the drawings
        self._map_viewer.clear()
        self._clear_preview_window()
        self._mod_geoms = None
        self._mod_latlon_locs.clear()
        self._mod_latlon_extents.clear()

        self.ui.tbl_primary.update()
        self.ui.tbl_secondary.update()

        new_p_row = self._get_ss_row_for_geom_idx(True, self._sel_geom_idx)
        self.ui.tbl_primary.selectRow(new_p_row)
        self._update_primary_selection()

        self._cur_sel_row_primary = 0
        self._cur_sel_row_secondary = 0

        if self._update_trigger == Triggers.TE_DELETE_PRIMARY:
            self._zoom_to_smallest_object()
            self._do_updates()

    def _on_delete_secondary(self):
        """Set the selected secondary geometry to be the selected primary."""
        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_DELETE_SECONDARY

        old_s_row = self._get_ss_row_for_geom_idx(False, self._sel_secondary_geom_idx)
        self._delete_geom(self._sel_secondary_geom_idx, True, True)

        # clear the drawings
        self._map_viewer.clear()
        self._clear_preview_window()
        self._mod_geoms = None
        self._mod_latlon_locs.clear()
        self._mod_latlon_extents.clear()

        self.ui.tbl_primary.update()
        self.ui.tbl_secondary.update()

        new_p_row = self._get_ss_row_for_geom_idx(True, self._sel_geom_idx)

        if new_p_row != self._cur_sel_row_primary:
            self.ui.tbl_primary.blockSignals(True)
            self.ui.tbl_primary.selectRow(new_p_row)
            self.ui.tbl_primary.blockSignals(False)
        self._update_primary_selection()

        self._sel_first_selectable_secondary_object(old_s_row)

        self._cur_sel_row_primary = 0
        self._cur_sel_row_secondary = 0

        if self._update_trigger == Triggers.TE_DELETE_SECONDARY:
            self._zoom_to_smallest_object()
            self._do_updates()

    def _on_accept_action(self):
        """Change the geometry."""
        if self._preview_is_displayed is False and self._sel_action != 'Delete' and not self._is_mpbd_tool:
            opt = message_box.message_with_n_buttons(self, 'It is recommended to preview the action before accepting.',
                                                     'SMS', ['Generate preview', 'Accept without previewing', 'Cancel'],
                                                     2, 2)
            if opt == 0:
                self._do_auto_updates = True
                self._do_updates()
                self._do_auto_updates = False
                return
            elif opt == 1:
                self._create_action_geoms()
            elif opt == 2:
                return

        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_ACCEPT_ACTION

        # stuff to help with selection afterwards
        self._cur_sel_row_primary = self.ui.tbl_primary.selectedIndexes()[0].row()

        # what is the action?
        if self._sel_action == 'Delete':
            self._accept_delete_action()
        elif self._sel_action == 'Concatenate':
            self._accept_concatenate_action()
        elif self._sel_action == 'Collapse':
            self._accept_collapse_action()
        elif self._mod_geoms is None:
            # nothing happened with the action
            return
        elif self._sel_action == 'Prune' or self._sel_action == 'Curvature Redistribution':
            self._accept_prune_or_redist()
        elif self._sel_action == 'Merge':
            self._accept_merge_polys_action()
        elif self._sel_action == 'Snap':
            self._accept_snap_action()

        # clear the drawings
        self._map_viewer.clear()
        self._clear_preview_window()
        self._mod_geoms = None
        self._mod_latlon_locs.clear()
        self._mod_latlon_extents.clear()

        self._sel_first_primary_object()

        if self._sel_action in self._actions_that_use_secondary:
            self._sel_first_selectable_secondary_object()

        self._cur_sel_row_primary = 0
        self._newest_geom_idx = None

        self._scroll_to_selection(True)
        self._scroll_to_selection(False)

        if self._update_trigger == Triggers.TE_ACCEPT_ACTION:
            self._zoom_to_smallest_object()
            self._do_updates()

    def _accept_delete_action(self):
        """Remove the item."""
        if self._sel_action not in self._multi_select_actions:
            self._delete_geom(self._sel_geom_idx, True, True)
        else:
            containing_set = set() if len(self._sel_geom_idx_multi) > 1 else None
            delete_group = set()
            for idx in self._sel_geom_idx_multi:
                self._delete_geom(idx, True, True, containing_set, set_to_delete_as_group=delete_group)

            if containing_set is not None:
                for c_poly in containing_set:
                    self._recreate_containing_polygon(c_poly, [], [])

            self._delete_set_from_ss(delete_group)

    def _accept_collapse_action(self):
        """Arc is collapsed and deleted - adjacent arcs are updated and polygons if necessary."""
        if (self._mod_geoms is None or len(self._mod_geoms) == 0):
            # same as a regular delete
            if self._msg_action is None:
                self._accept_delete_action()
            return

        # delete the collapsed arc
        self._delete_geom(self._sel_geom_idx, False)

        # modify the attached arcs
        for i, mod_arc in enumerate(self._mod_geoms):
            geom_idx = self._addtl_mod_geom_list_idxs[i]
            latlon = self._mod_latlon_locs[i]
            extents = self._mod_latlon_extents[i]
            self._modify_arc(geom_idx, mod_arc, latlon, extents)

    def _recreate_containing_polygon(self, c_poly, remove_holes, add_holes):
        """Arcs in this polygon have been updated, so recreate it for display."""
        if c_poly in self._deleted_geom_idxs or c_poly in self._is_deleting:
            # this shouldn't happen?
            return

        # make a list of the original holes and the new ones to add
        holes = self._get_hole_polys(c_poly) + add_holes
        holes = [hole for hole in holes if (hole not in remove_holes and hole not in self._deleted_geom_idxs)]

        loop_locs = [gsh.perim_pts_from_sh_poly(self._geoms[hole]) for hole in holes]
        loop_arc_lists = [self._get_poly_outer_arcs(hole) for hole in holes]

        for hole_poly_idx in add_holes:
            self._set_containing_poly(hole_poly_idx, c_poly)

        outer_locs = gsh.perim_pts_from_sh_poly(self._geoms[c_poly])
        new_poly = shPolygon(outer_locs, loop_locs)
        mod_geom_latlon_locs, mod_latlon_extents = self._transform_geom(new_poly)
        self._update_geom(c_poly, new_poly, mod_geom_latlon_locs, mod_latlon_extents)

        # update the dataframe
        self._set_poly_inner_arc_lists(c_poly, loop_arc_lists)
        self._set_poly_hole_polys(c_poly, holes)

    def _accept_prune_or_redist(self):
        """Arc is updated - polygons if necessary."""
        # modify the arcs
        self._modify_arc(self._sel_geom_idx, self._mod_geoms[0], self._mod_latlon_locs[0],
                         self._mod_latlon_extents[0])

    def _accept_snap_action(self):
        """Arc is updated."""
        if self._mod_geoms and len(self._mod_geoms) > 0:
            did_split = len(self._mod_geoms) == 2

            orig_arc_idx = self._sel_geom_idx

            # modify the first arc
            self._modify_arc(orig_arc_idx, self._mod_geoms[0], self._mod_latlon_locs[0], self._mod_latlon_extents[0])

            if did_split:
                # if we are here, it split the arc, so add the second arc
                self._add_geom(self._mod_geoms[1], self._mod_latlon_locs[1], self._mod_latlon_extents[1])

    def _accept_concatenate_action(self):
        """Arc is updated - polygons if necessary."""
        if self._mod_geoms is None or len(self._mod_geoms) == 0:
            return

        # modify the primary arc
        self._modify_arc(self._sel_geom_idx, self._mod_geoms[0], self._mod_latlon_locs[0],
                         self._mod_latlon_extents[0])

        # delete the secondary arc
        self._delete_geom(self._sel_secondary_geom_idx)

    def _accept_merge_polys_action(self):
        """Keep arcs as well as possible, create new polygons."""
        delete_group = set()
        p_idx = self._sel_geom_idx
        s_idx = self._sel_secondary_geom_idx
        new_poly_idxs = []

        # first figure out which arcs need to be modified and which ones need to be created
        all_arcs = self._get_all_arcs_from_poly(p_idx).union(self._get_all_arcs_from_poly(s_idx))
        arc_nodes = dict()
        for arc in all_arcs:
            node_locs = gsh.endpt_locs_from_ls(self._geoms[arc])
            for node in node_locs:
                if node not in arc_nodes.keys():
                    arc_nodes[node] = set()
                arc_nodes[node].add(arc)

        # get all polygons that are valid for us to modify (primary, secondary, holes contained by p and s, and polys
        # that contain p and s
        allowed_to_modify = ([p_idx, s_idx] + list(self._removed_inner_polys) + self._get_hole_polys(p_idx))
        allowed_to_modify.extend(self._get_hole_polys(s_idx))

        contains_p = self._get_containing_poly(p_idx)
        if contains_p:
            allowed_to_modify.append(contains_p)
        contains_s = self._get_containing_poly(s_idx)
        if contains_s:
            allowed_to_modify.append(contains_s)

        used_arcs = []
        modified_arcs = []
        polys_created_from_merge = self._mod_geoms.copy()
        tmp_ss_dict = {   # DataFrame of results data
            STR_DISP_TYPE: [],
            STR_ID: [],
            STR_NUM_ARCS: [],
            STR_AREA: [],
            STR_PERIM: [],
            STR_DIST_TO_P: [],
            STR_GEOM_IDX: []
        }
        tmp_geom_dict = {
            'geom_idx': [],
            'feature_id': [],  # id that will show up in SMS
            'a_outer_polys': [],  # polygons that use this arc as an outer boundary
            'a_inner_polys': [],  # polygons that use this arc as an internal boundary
            'p_outer_arcs': [],  # outer arcs that make up the perimeter of this polygon
            'p_inner_arcs': [],  # inner arcs that make up interior boundaries of this polygon
            'p_containing_poly': [],  # the polygon that contains this polygon as a hole
            'p_hole_polys': []  # polygons that are holes in this polygon
        }

        for i, merged_poly in enumerate(polys_created_from_merge):
            merged_poly_locs = gsh.pts_from_shapely_poly_tuple(merged_poly)

            # do the perimeter first
            # do the perimeter first
            perimeter = merged_poly_locs[0]
            outer_arc_list = self._create_arcs_used_for_loop(perimeter, arc_nodes, [] if i > 0 else used_arcs,
                                                             modified_arcs, allowed_to_modify, tmp_ss_dict,
                                                             tmp_geom_dict)

            holes_list = []
            if len(merged_poly_locs) > 1:
                # do the holes
                for hole in merged_poly_locs[1:]:
                    holes_list.append(self._create_arcs_used_for_loop(hole, arc_nodes, used_arcs, modified_arcs,
                                                                      allowed_to_modify, tmp_ss_dict, tmp_geom_dict))

            new_poly_idx = self._add_geom(merged_poly, self._mod_latlon_locs[i], self._mod_latlon_extents[i],
                                          len(outer_arc_list), tmp_ss_dict, tmp_geom_dict)
            new_poly_idxs.append(new_poly_idx)

            self._set_poly_outer_arcs(new_poly_idx, outer_arc_list, tmp_geom_dict)
            self._set_poly_inner_arc_lists(new_poly_idx, holes_list, tmp_geom_dict)

        # now that we are done adding new geometries, add everything to the dataframes
        self._ss_df = pd.concat([self._ss_df, pd.DataFrame(tmp_ss_dict, columns=self._ss_df.columns)],
                                ignore_index=True)
        self._geom_df = pd.concat([self._geom_df, pd.DataFrame(tmp_geom_dict, columns=self._geom_df.columns)],
                                  ignore_index=True)

        # map over the holes
        all_holes = set(self._get_hole_polys(p_idx)).union(set(self._get_hole_polys(s_idx)))
        hole_polys_in_merged_poly = \
            [h for h in all_holes if (h not in self._removed_inner_polys and h not in self._deleted_geom_idxs)]

        # if we were doing a merge by difference, but only holes were merged, we need to make sure we include the
        # merged hole poly as a "hole" in main poly
        if self._merge_is_difference is True and len(new_poly_idxs) == 2:
            hole_polys_in_merged_poly.append(new_poly_idxs[1])

        self._set_poly_hole_polys(new_poly_idxs[0], hole_polys_in_merged_poly)

        for hole_arcs in holes_list:
            for hole_arc in hole_arcs:
                self._assign_inner_poly_to_arc(hole_arc, new_poly_idxs[0])

        for hole_poly in hole_polys_in_merged_poly:
            self._set_containing_poly(hole_poly, new_poly_idxs[0])

        # update affected polys
        self._is_deleting = [p_idx, s_idx] + self._removed_inner_polys

        # delete unused arcs that are not part of polys we can't modify
        to_delete = []
        for arc_idx in all_arcs:
            if arc_idx in used_arcs:
                continue
            polys_using_arc = self._get_all_polys_using_arc(arc_idx)
            not_allowed_polys = [idx for idx in polys_using_arc if idx not in allowed_to_modify]
            if len(not_allowed_polys) == 0:
                to_delete.append(arc_idx)

        # we are only recreating outer (containing) polygons
        if contains_p and contains_s and contains_p == contains_s:
            holes_to_remove = [p_idx, s_idx] + self._removed_inner_polys
            self._recreate_containing_polygon(contains_p, holes_to_remove, new_poly_idxs)

        # delete unused arcs?
        for delete_arc in to_delete:
            self._delete_geom(delete_arc, set_to_delete_as_group=delete_group)

        # delete original polygons
        self._delete_geom(p_idx, set_to_delete_as_group=delete_group, recreate_c_poly=False)
        self._delete_geom(s_idx, set_to_delete_as_group=delete_group, recreate_c_poly=False)

        # delete any holes
        for delete_me in self._removed_inner_polys:
            self._delete_geom(delete_me, set_to_delete_as_group=delete_group, delete_attached_arcs=True)

        # remove everything from the ss
        self._delete_set_from_ss(delete_group)

        self._newest_geom_idx = new_poly_idxs[0] if len(new_poly_idxs) > 0 else None

        self.p_model.update_data_frame(self._ss_df)
        return new_poly_idxs

    def _create_arcs_used_for_loop(self, loop, arc_nodes_dict, used_arcs, modified_arcs, allowed_to_modify,
                                   tmp_ss_dict, tmp_geom_dict):
        """Find out which arcs the loop points belong to and recreate the arcs as close to the original as possible.

        Args:
            loop (list): List of points that make up the loop
            arc_nodes_dict (dict): Nodes that were originally part of the polygons and the arcs they go with
            used_arcs (list): Running list of arcs that have already been used (so we can't use them again)
            modified_arcs (list): List of arcs that have been modified from their original state
            allowed_to_modify (list): List of polygon idxs. If an arc is used by a polygon not in this list, we are
                                      not allowed to modify it because it gets too ugly
            tmp_ss_dict (dict): Temporary dictionary to be concatenated with the dataframes later
            tmp_geom_dict (dict): Temporary dictionary to be concatenated with the dataframes later
        """
        # get rid of the repeated point at the end
        loop.pop()

        new_arcs = []

        # which nodes are found in this loop
        all_orig_nodes = set(arc_nodes_dict.keys())
        node_idxs = [idx for idx, loc in enumerate(loop) if loc in all_orig_nodes]

        if len(node_idxs) == 0:
            # we get to do this as one arc?
            loop.append(loop[0])
            new_arc = LineString(loop)
            # latlon, extents = self._transform_geom(new_arc)
            new_arc_idx = self._add_geom(new_arc, [], [], tmp_ss=tmp_ss_dict, tmp_geom=tmp_geom_dict)
            used_arcs.append(new_arc_idx)
            return [new_arc_idx]

        for i, idx in enumerate(node_idxs):
            next_i = i + 1 if i < len(node_idxs) - 1 else 0
            next_idx = node_idxs[next_i]
            if next_idx > idx:
                new_arc_pts = loop[idx:next_idx + 1]
            else:
                new_arc_pts = loop[idx:] + loop[:next_idx + 1]

            # create the geometry item
            new_arc = LineString(new_arc_pts)
            # latlon, extents = self._transform_geom(new_arc)
            arc_idx = -1
            existing_arc_pts = []
            arc_is_modified = True

            start, end = loop[idx], loop[next_idx]
            arcs_from_start = arc_nodes_dict[start]
            arcs_from_end = arc_nodes_dict[end]

            arcs_to_use = arcs_from_start.intersection(arcs_from_end)
            arcs_to_use = {arc for arc in arcs_to_use if arc not in used_arcs}

            if len(arcs_to_use) > 0:
                arc_idx = arcs_to_use.pop()

            if arc_idx == -1:
                next_pt = new_arc_pts[1]

                # get arc points from start
                for arc in arcs_from_start:
                    if arc in used_arcs:
                        continue
                    existing_arc_pts = gsh.pts_from_ls_tuple(self._geoms[arc])
                    if next_pt in existing_arc_pts:
                        arc_idx = arc
                        break
                # get arc point from end
                if arc_idx == -1:
                    next_end_pt = new_arc_pts[-2]
                    for arc in arcs_from_end:
                        if arc in used_arcs:
                            continue
                        existing_arc_pts = gsh.pts_from_ls_tuple(self._geoms[arc])
                        if next_end_pt in existing_arc_pts:
                            arc_idx = arc
                            break

            if arc_idx == -1:
                new_arc_idx = self._add_geom(new_arc, [], [], tmp_ss=tmp_ss_dict, tmp_geom=tmp_geom_dict)
            else:
                if len(existing_arc_pts) == 0:
                    existing_arc_pts = gsh.pts_from_ls_tuple(self._geoms[arc_idx])
                if existing_arc_pts == new_arc_pts:
                    arc_is_modified = False
                else:
                    existing_arc_pts.reverse()
                    if existing_arc_pts == new_arc_pts:
                        arc_is_modified = False
                if arc_is_modified:
                    polys_using_arc = self._get_all_polys_using_arc(arc_idx)
                    not_allowed_polys = [poly_idx for poly_idx in polys_using_arc if poly_idx not in allowed_to_modify]
                    if len(not_allowed_polys) == 0:
                        self._modify_arc(arc_idx, new_arc, [], [])
                        modified_arcs.append(arc_idx)
                        new_arc_idx = arc_idx
                    else:
                        # put this arc in "used" so that we don't try to use it later - we can't
                        used_arcs.append(arc_idx)
                        new_arc_idx = self._add_geom(new_arc, [], [], tmp_ss=tmp_ss_dict,
                                                     tmp_geom=tmp_geom_dict)
                else:
                    new_arc_idx = arc_idx

            new_arcs.append(new_arc_idx)
            used_arcs.append(new_arc_idx)

        return new_arcs

    def _modify_arc(self, geom_idx, mod_arc, latlon, extents):
        """Update node assignments arc props.

        Args:
            geom_idx (idx): Index of arc to modify
            mod_arc (LineString): Shapely LineString of modified arc
            latlon (list): Latlon locations of modified arc
            extents (list): Extents of arc
        """
        orig_end_pts = gsh.endpt_locs_from_ls(self._geoms[geom_idx])
        new_end_pts = gsh.endpt_locs_from_ls(mod_arc)

        was_loop = orig_end_pts[0] == orig_end_pts[1]
        if was_loop:
            orig_loop_pt_remains = orig_end_pts[0] in new_end_pts

        if orig_end_pts[0] != new_end_pts[0]:
            if was_loop and orig_loop_pt_remains:
                self._add_endpoint_loc(geom_idx, new_end_pts[0])
            else:
                self._update_endpoint_loc(geom_idx, orig_end_pts[0], new_end_pts[0])
        if orig_end_pts[1] != new_end_pts[1]:
            if was_loop and orig_loop_pt_remains:
                self._add_endpoint_loc(geom_idx, new_end_pts[1])
            else:
                self._update_endpoint_loc(geom_idx, orig_end_pts[1], new_end_pts[1])

        self._update_geom(geom_idx, mod_arc, latlon, extents)

    def _update_endpoint_loc(self, geom_idx, orig_end, new_end):
        """Update node assignments arc props.

        Args:
            geom_idx (idx): Index of arc to modify
            orig_end (list): Location of original endpoint
            new_end (list): Location of new endpoint
        """
        self._remove_arc_from_endpoint_loc(geom_idx, orig_end)
        if new_end not in self._node_pts_to_arc_lst_idxs.keys():
            self._node_pts_to_arc_lst_idxs[new_end] = set()
        self._node_pts_to_arc_lst_idxs[new_end].add(geom_idx)

    def _add_endpoint_loc(self, geom_idx, end_pt):
        """Update node assignments arc props.

        Args:
            geom_idx (idx): Index of arc to modify
            end_pt (list): Location of endpoint
        """
        if end_pt not in self._node_pts_to_arc_lst_idxs.keys():
            self._node_pts_to_arc_lst_idxs[end_pt] = set()
        self._node_pts_to_arc_lst_idxs[end_pt].add(geom_idx)

    def _remove_arc_from_endpoint_loc(self, geom_idx, end_pt):
        """Update node assignments arc props.

        Args:
            geom_idx (idx): Index of arc to modify
            end_pt (list): Location of endpoint
        """
        if end_pt in self._node_pts_to_arc_lst_idxs.keys():
            arcs_at_node = self._node_pts_to_arc_lst_idxs[end_pt]
            if geom_idx in arcs_at_node:
                self._node_pts_to_arc_lst_idxs[end_pt].remove(geom_idx)
                for other_arc in arcs_at_node:
                    self._update_num_connected_arcs(other_arc)

                if len(arcs_at_node) == 0:
                    del self._node_pts_to_arc_lst_idxs[end_pt]

    def _on_change_action_parameter(self):
        """Recreate the drawings when an action parameter is changed."""
        if self._update_trigger == Triggers.TE_NONE:
            self._do_updates()

    def _on_chk_display_map(self):
        """Recreate the drawings with or without the background image."""
        self._update_map_view = True
        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_DISPLAY_MAP
            self._do_updates(False)

    def _on_chk_fill_polys(self):
        """Recreate the drawings with or without the background image."""
        self._update_map_view = True
        if self._update_trigger == Triggers.TE_NONE:
            self._update_trigger = Triggers.TE_FILL_POLYS
            self._do_updates(False)

    def _update_prop_for_geom_idx(self, geom_idx, property, val):
        """Update geometry data.

        Args:
            geom_idx (int): geometry list index of geometry to update
            property (str): attribute to update
            val (float): updated value
        """
        if property == STR_DIST_TO_P:
            # this is the only case where we need it to be the secondary df
            self._secondary_ss_df.loc[self._secondary_ss_df[STR_GEOM_IDX] == geom_idx, property] = val
        self._ss_df.loc[self._ss_df[STR_GEOM_IDX] == geom_idx, property] = val

    def _update_geom(self, geom_idx, mod_geom, geom_locs_latlon, latlon_extents):
        """Update geometry after performing an action.

        Args:
            geom_idx (int): index of geometry
            mod_geom (Shapely obj): modified geometry
            geom_locs_latlon (list): modified geometry locations in latlon
            latlon_extents (list): extents in latlon
        """
        old_geom = self._geoms[geom_idx]
        self._geoms[geom_idx] = mod_geom
        self._geom_latlon_locs[geom_idx] = geom_locs_latlon
        self._latlon_extents[geom_idx] = deepcopy(latlon_extents)

        if self._geom_type_shapely != mod_geom.geom_type:
            return

        if self._is_geo:
            geom_for_props = self._convert_latlon_geom_to_utm(mod_geom)
        else:
            geom_for_props = mod_geom
        self._geoms_for_props[geom_idx] = geom_for_props

        if mod_geom.geom_type == 'LineString':
            self._update_prop_for_geom_idx(geom_idx, STR_LEN, geom_for_props.length)
            self._update_prop_for_geom_idx(geom_idx, STR_NUM_SEGS, len(mod_geom.xy[0]) - 1)
            self._update_num_connected_arcs(geom_idx)
        elif mod_geom.geom_type == 'Polygon':
            self._update_prop_for_geom_idx(geom_idx, STR_NUM_ARCS, 1)
            self._update_prop_for_geom_idx(geom_idx, STR_AREA, geom_for_props.area)
            self._update_prop_for_geom_idx(geom_idx, STR_PERIM, geom_for_props.exterior.length)

        # modify the rtree - delete the old
        old_box = old_geom.bounds
        self._rtree.delete(geom_idx, old_box)
        # set the new
        box = mod_geom.bounds
        self._rtree.insert(geom_idx, box, obj=box)

        # update the simplified locs if necessary
        if self._get_prop_for_geom_idx(STR_DISP_TYPE, geom_idx) == 'Simplified':
            self._get_simplified_locs(geom_idx, update=True)

    def _add_geom(self, new_geom, new_latlon_locs, new_latlon_extents, num_perim_arcs=0, tmp_ss=None, tmp_geom=None):
        """Add a new geometry to the data frame.

        Args:
            new_geom (object): New geometry to add.
            new_latlon_locs (list): New geometry locations in latlon for display.
            new_latlon_extents (list): List of extents in latlon.
            num_perim_arcs (int): Number of arcs that make up the polygon perimeter.
            tmp_ss (dict): Temporary dictionary that will be used to create DataFrame.
            tmp_geom (dict) Temporary dictionary that will be used to create DataFrame.

        Returns:
            new_geom_idx (int): Index of the new item.
        """
        geom_idx = self._new_geom_idx
        self._new_geom_idx += 1
        self._geoms.append(new_geom)
        self._geom_latlon_locs.append(new_latlon_locs)
        self._latlon_extents.append(deepcopy(new_latlon_extents))

        if self._is_geo:
            geom_for_props = self._convert_latlon_geom_to_utm(new_geom)
        else:
            geom_for_props = new_geom
        self._geoms_for_props.append(geom_for_props)

        num_connections = 0
        if new_geom.geom_type == 'LineString':
            # add the nodes
            end_pts = gsh.endpt_locs_from_ls(new_geom)
            for end_pt in end_pts:
                self._add_endpoint_loc(geom_idx, end_pt)
                arcs_at_loc = self._node_pts_to_arc_lst_idxs[end_pt]
                num_connections += len(arcs_at_loc) - 1

        if tmp_ss is None and tmp_geom is None and new_geom.geom_type == 'LineString':
            # we don't add polygons without the tmp dataframes, so it should only be linestrings here
            # order of ls props is: display, ID, length, num segments, num connected arcs, dist to primary, geom idx
            self.tool._max_arc_id += 1
            props = ['Normal', self.tool._max_arc_id, geom_for_props.length, len(geom_for_props.xy[0]) - 1,
                     num_connections, -1, geom_idx]
            arcs_polys_df = [geom_idx, self.tool._max_arc_id, [], [], None, None, None, None]

            # update both data frames
            self._ss_df = pd.concat([self._ss_df, pd.DataFrame([props], columns=self._ss_df.columns)],
                                    ignore_index=True)
            self.p_model.update_data_frame(self._ss_df)

            self._geom_df = pd.concat([self._geom_df, pd.DataFrame([arcs_polys_df], columns=self._geom_df.columns)],
                                      ignore_index=True)
        else:
            if new_geom.geom_type == 'LineString':
                self.tool._max_arc_id += 1

                tmp_geom['geom_idx'].append(geom_idx)
                tmp_geom['feature_id'].append(self.tool._max_arc_id)
                tmp_geom['a_outer_polys'].append([])
                tmp_geom['a_inner_polys'].append([])
                tmp_geom['p_outer_arcs'].append(None)
                tmp_geom['p_inner_arcs'].append(None)
                tmp_geom['p_containing_poly'].append(None)
                tmp_geom['p_hole_polys'].append([])
            else:
                self.tool._max_poly_id += 1
                if self._geom_type == 'Polygon':
                    tmp_ss[STR_DISP_TYPE].append('Normal')
                    tmp_ss[STR_ID].append(self.tool._max_poly_id)
                    tmp_ss[STR_NUM_ARCS].append(num_perim_arcs)
                    tmp_ss[STR_AREA].append(geom_for_props.area)
                    tmp_ss[STR_PERIM].append(geom_for_props.exterior.length)
                    tmp_ss[STR_DIST_TO_P].append(-1.0)
                    tmp_ss[STR_GEOM_IDX].append(geom_idx)

                tmp_geom['geom_idx'].append(geom_idx)
                tmp_geom['feature_id'].append(self.tool._max_poly_id)
                tmp_geom['a_outer_polys'].append(None)
                tmp_geom['a_inner_polys'].append(None)
                tmp_geom['p_outer_arcs'].append([])
                tmp_geom['p_inner_arcs'].append([])
                tmp_geom['p_containing_poly'].append(None)
                tmp_geom['p_hole_polys'].append([])

        if new_geom.geom_type == self._geom_type_shapely:
            # add to the rtree
            box = new_geom.bounds
            self._rtree.insert(geom_idx, box, obj=box)

        return geom_idx

    def _delete_set_from_ss(self, delete_group):
        """Delete a geometry item.

        Args:
            delete_group (set): geometry idxs to delete from ss
        """
        remove = [idx for idx in delete_group if self._geoms[idx].geom_type == self._geom_type_shapely]

        self._ss_df = pd.DataFrame(self._ss_df.loc[~self._ss_df[STR_GEOM_IDX].isin(remove)])
        self._ss_df.reset_index(drop=True, inplace=True)
        self.p_model.update_data_frame(self._ss_df)

        for del_idx in remove:
            del_geom = self._geoms[del_idx]
            del_box = del_geom.bounds
            self._rtree.delete(del_idx, del_box)

        for del_idx in delete_group:
            self._geoms[del_idx] = []
            self._geom_latlon_locs[del_idx] = []
            self._geoms_for_props[del_idx] = []

    def _delete_geom(self, geom_list_idx, delete_attached_polys=False, delete_attached_arcs=False,
                     set_containing_polys=None, set_to_delete_as_group=None, recreate_c_poly=True):
        """Delete a geometry item.

        Args:
            geom_list_idx (int): geometry id to delete
            delete_attached_polys (bool): True if we should delete the polygons using the arcs.
            delete_attached_arcs (bool): True if we should delete arcs from the polygon.
            set_containing_polys (set): Set of polys to recreate if we are doing multiple deletes - otherwise just
                                        handle it here
            set_to_delete_as_group (set): Set of geom_idxs to remove from the df - do them all at once
            recreate_c_poly (bool): True if we should recreate the containing polygon
        """
        if geom_list_idx in self._deleted_geom_idxs:  # already deleted
            return

        self._deleted_geom_idxs.append(geom_list_idx)

        del_geom = self._geoms[geom_list_idx]
        if del_geom.geom_type == 'LineString':
            end_pts = gsh.endpt_locs_from_ls(del_geom)
            self._remove_arc_from_endpoint_loc(geom_list_idx, end_pts[0])
            self._remove_arc_from_endpoint_loc(geom_list_idx, end_pts[1])

            all_polys = self._get_all_polys_using_arc(geom_list_idx)
            if len(all_polys) > 0:
                outer_polys = self._get_polys_from_outer_arc(geom_list_idx)
                for outer_poly in outer_polys:
                    # if delete_attached_polys:
                    #     self._delete_geom(outer_poly, set_to_delete_as_group=set_to_delete_as_group)
                    # else:
                    self._remove_outer_arc_from_poly(geom_list_idx, outer_poly)
                containing_polys = self._get_polys_from_inner_arc(geom_list_idx)
                for c_poly in containing_polys:
                    self._remove_inner_arc_from_poly(geom_list_idx, c_poly)
        else:
            outer_arcs = self._get_poly_outer_arcs(geom_list_idx)
            c_poly = self._get_containing_poly(geom_list_idx)

            inner_arc_lists = self._get_poly_inner_arc_lists(geom_list_idx)
            for inner_arc_list in inner_arc_lists:
                for inner_arc in inner_arc_list:
                    self._remove_inner_poly_from_arc(inner_arc, geom_list_idx)
                    polys_using_arc = self._get_all_polys_using_arc(inner_arc)
                    # if len(polys_using_arc) == 0 and delete_attached_arcs:
                    #     self._delete_geom(inner_arc, set_to_delete_as_group=set_to_delete_as_group)

            for outer_arc in outer_arcs:
                self._remove_outer_poly_from_arc(outer_arc, geom_list_idx)
                polys_using_arc = self._get_all_polys_using_arc(outer_arc)
                if delete_attached_arcs:
                    if len(polys_using_arc) == 0 or (len(polys_using_arc) == 1 and c_poly in polys_using_arc):
                        self._delete_geom(outer_arc, set_to_delete_as_group=set_to_delete_as_group)

            # is it a hole in a polygon?
            if c_poly is not None:
                self._remove_hole_from_poly(c_poly, geom_list_idx)
                if set_containing_polys is not None:
                    set_containing_polys.add(c_poly)
                elif recreate_c_poly:
                    self._recreate_containing_polygon(c_poly, [], [])

            # does it contain a hole?
            hole_polys = self._get_hole_polys(geom_list_idx)
            for hole in hole_polys:
                self._remove_containing_poly_from_poly(hole, geom_list_idx)

        if set_to_delete_as_group is None:
            # delete as a set of 1
            self._delete_set_from_ss(set([geom_list_idx]))
        else:
            set_to_delete_as_group.add(geom_list_idx)

    def _clear_preview_window(self):
        """Clear the preview window drawing."""
        clear_geometry(self._url_action_preview_file)
        self._action_preview.setUrl(QUrl.fromLocalFile(self._url_action_preview_file))
        self._action_preview.show()

    def _do_collapse_arc(self, primary, keep_start):
        """Collapse an arc, keeping the specified node.

        Args:
            primary (LineString): LineString to collapse_
            keep_start (bool): True for keeping the start node, False for the end
        Returns:
            (list): modified adjacent arcs
            (list): modified arc ids
        """
        ignore_idxs = list(self._deleted_geom_idxs) + [self._sel_geom_idx]
        end_pts = gsh.endpt_locs_from_ls(primary)
        conn_at_start_idxs = self._node_pts_to_arc_lst_idxs[end_pts[0]]
        conn_at_end_idxs = self._node_pts_to_arc_lst_idxs[end_pts[1]]

        # geometry indices if connect arcs that are not already deleted or the primary arc
        conn_at_start_idxs = [idx for idx in conn_at_start_idxs if idx not in ignore_idxs]
        conn_at_end_idxs = [idx for idx in conn_at_end_idxs if idx not in ignore_idxs]

        conn_at_start = [self._geoms[idx] for idx in conn_at_start_idxs]
        conn_at_end = [self._geoms[idx] for idx in conn_at_end_idxs]

        mod_geoms, self._msg_action = collapse_arc(primary, keep_start, conn_at_start, conn_at_end)

        if mod_geoms:
            self._addtl_mod_geom_list_idxs = conn_at_end_idxs if keep_start else conn_at_start_idxs

        return mod_geoms

    def _do_snap_arc(self):
        """Snap primary arc to the secondary arc.

        Args:
            primary (LineString): LineString to snap
            secondary (LineString): LineString to snap primary to
        Returns:
            (list): modified adjacent arcs
            (list): modified arc ids
        """
        result = [self._sel_geom_item]
        if self._sel_secondary_geom_idx is not None:
            non_geo_primary = self._geoms_for_props[self._sel_geom_idx]
            non_geo_secondary = self._geoms_for_props[self._sel_secondary_geom_idx]
            self._snap_tol = float(self.ui.edt_action.text())
            result = snap_arcs(non_geo_primary, non_geo_secondary, self._snap_tol)

            if result and self._is_geo:
                result = [LineString(self._reproject_pts_to_latlon(gsh.pts_from_ls(arc), True)) for arc in result]
        return result

    def _do_prune_arc(self):
        """Prune the primary arc.

        Args:

        Returns:
            (list): modified adjacent arcs
            (list): modified arc ids
        """
        prune_params = {}
        if self.ui.chk_advanced.isChecked():
            self._join_type = self.ui.cbx_action_options_2.currentText()
            self._mitre_limit = None if not self.ui.chk_action_2.isChecked() else float(self.ui.edt_action_2.text())
            self._debug_arcs = self.ui.cbx_action_options_3.currentText()
            self._num_steps = int(self.ui.edt_action_3.text())

            prune_params['join'] = [self._join_type]
            prune_params['mitre_limit'] = [self._mitre_limit]
            prune_params['num_steps'] = [self._num_steps]
            prune_params['debug_arcs'] = [self._debug_arcs]
            self._msg_info = None
            self._show_hide_messages()
        else:
            prune_params['join'] = ['mitre', 'round', 'bevel', 'mitre']
            prune_params['mitre_limit'] = [100.0, None, None, 100.0]
            prune_params['num_steps'] = [5, 5, 5, 10]
            prune_params['debug_arcs'] = ['none', 'none', 'none', 'none']
        self._prune_distance = float(self.ui.edt_action.text())

        # we want a non-geographic arc if we are using a specified distance since they don't match up
        non_geo_arc = self._geoms_for_props[self._sel_geom_idx]

        num_tries = len(prune_params['join'])
        for i in range(num_tries):
            join = prune_params['join'][i]
            mitre_limit = prune_params['mitre_limit'][i]
            num_steps = prune_params['num_steps'][i]
            debug_arcs = prune_params['debug_arcs'][i]
            if num_tries > 1:
                if join != 'mitre':
                    self._msg_info = f'Attempt {i + 1} of 5: {join} join method'
                    self._msg_action = None
                else:
                    self._msg_info = f'Attempt {i + 1} of 5: mitre join method, {mitre_limit} mitre limit'
                    self._msg_action = None
                self._show_hide_messages()
            pruned_arc, d_arcs, self._msg_action = prune_arc(self._sel_geom_item, non_geo_arc, self._prune_distance,
                                                             self.ui.cbx_action_options.currentText() == 'Left',
                                                             join, num_steps, debug_arcs, mitre_limit)
            if pruned_arc is not None:
                if num_tries > 1:
                    if join != 'mitre':
                        self._msg_info = f'Pruned using the {join} join method.'
                    else:
                        self._msg_info = f'Pruned using the mitre join method and a {mitre_limit} mitre limit.'
                break

        if pruned_arc is None:
            self._msg_info = None

        if len(d_arcs) > 0 and self._is_geo:
            # debug arcs aren't in geographic
            d_arcs = [LineString(self._reproject_pts_to_latlon(gsh.pts_from_ls(arc), True)) for arc in d_arcs]

        if pruned_arc and len(gsh.pts_from_ls(pruned_arc)) == len(gsh.pts_from_ls(self._sel_geom_item)):
            self._msg_action = 'Action result: No impact'

        if pruned_arc:
            return [pruned_arc], d_arcs

        return None, d_arcs

    def _do_concatenate_arcs(self):
        """Combine arcs if they share a node."""
        if self._sel_secondary_geom_idx is None:
            self._msg_action = 'A secondary arc must be selected.'
            return None

        p_arc = self._geoms[self._sel_geom_idx]
        p_ends = gsh.endpt_locs_from_ls(p_arc)
        if p_ends[0] == p_ends[1]:
            self._msg_action = 'The primary arc is a loop. Cannot concatenate.'
            return None

        s_arc = self._geoms[self._sel_secondary_geom_idx]
        s_ends = gsh.endpt_locs_from_ls(s_arc)
        if s_ends[0] == s_ends[1]:
            self._msg_action = 'The secondary arc is a loop. Cannot concatenate.'
            return None

        p_pts = gsh.pts_from_ls(p_arc)
        s_pts = gsh.pts_from_ls(s_arc)
        if p_ends[0] == s_ends[0]:
            shared_node = p_ends[0]
            s_pts.reverse()
            new_pts = s_pts + p_pts[1:]
        elif p_ends[0] == s_ends[1]:
            shared_node = p_ends[0]
            new_pts = s_pts + p_pts[1:]
        elif p_ends[1] == s_ends[0]:
            shared_node = p_ends[1]
            new_pts = p_pts + s_pts[1:]
        elif p_ends[1] == s_ends[1]:
            shared_node = p_ends[1]
            s_pts.reverse()
            new_pts = p_pts + s_pts[1:]
        else:
            self._msg_action = 'Selected arcs do not share a node.'
            return None

        num_connected_arcs = len(self._node_pts_to_arc_lst_idxs[shared_node])
        if num_connected_arcs > 2:
            self._msg_action = 'More than two arcs share this node. Cannot concatenate.'
            return None

        return [LineString(new_pts)]

    def _do_curvature_redist(self):
        """Perform curvature redistribution."""
        non_geo_arc = self._geoms_for_props[self._sel_geom_idx]
        mod_arc, self._msg_action = curvature_redistribution(non_geo_arc, self._max_delta, self._min_seg_length)
        if self._is_geo and mod_arc:
            mod_arc = LineString(self._reproject_pts_to_latlon(gsh.pts_from_ls(mod_arc), True))

        return [mod_arc] if mod_arc else None

    def _do_updates(self, recreate_action_items=True):
        """Redraw."""
        self._draw_preview = self._do_auto_updates or self._update_trigger == Triggers.TE_CREATE_PREVIEW
        if not self._draw_preview:
            zoom_triggers = [Triggers.TE_ZOOM_TO_PRIMARY, Triggers.TE_ZOOM_TO_SECONDARY,
                             Triggers.TE_FILL_POLYS, Triggers.TE_DISPLAY_MAP]
            self._draw_preview = self._preview_is_displayed and self._update_trigger in zoom_triggers

        self._preview_is_displayed = False
        if recreate_action_items and self._draw_preview:
            self._create_action_geoms()
        self._update_viewers()

        # reset
        self._update_trigger = Triggers.TE_NONE

        self._show_hide_messages()

    def _update_viewers(self):
        """Create the url file for displaying the geometry."""
        if self._allow_display_update is False:
            return

        # reset everything
        if self._update_map_view:
            folium.LayerControl().reset()

        # action, primary, selected secondary, secondary, debug
        weights = [3, 3, 2, 1, 1, 1]
        dash = [None, None, None, None, None, '5, 8']
        sel_secondary_geom_ids = []

        action_ids_to_ignore = [self._get_prop_for_geom_idx(STR_ID, g_idx) for g_idx in self._addtl_mod_geom_list_idxs]

        if self.ui.chk_fill_polys.isChecked() is False:
            fill_colors = [None, None, None, None, None, None]
            opacity = [None, None, None, None, None, None]
        else:
            opacity = [0.2, 0.2, 0.2, None, None, None]
            fill_colors = [self._display_colors[0], self._display_colors[1], self._display_colors[2], None, None, None]

        all_geoms = [self._action_display, self._primary_display, self._sel_secondary_display, self._secondary_display,
                     self._debug_arcs_display, self._primary_display]

        # if self._draw_preview and self._display_dashed_item:
        #     # if we have debug arcs, don't also display the dashed original item
        #     self._display_dashed_item = len(self._debug_arcs_display) == 0

        action_geoms = []
        map_geoms = []
        for type, geom_group in enumerate(all_geoms):
            if geom_group is None or len(geom_group) == 0:
                continue
            if type == 0 and self._draw_preview is False:
                continue
            if type == 1 and self._update_map_view is False:
                continue  # not drawing this group
            if type == 5 and self._display_dashed_item is False:
                continue  # not drawing the original dashed item
            for geom in geom_group:
                geom_id = geom[1]
                # the selected secondary geometries will also be in the secondary list - don't draw them twice
                if type == 2:
                    sel_secondary_geom_ids.append(geom_id)
                elif type == 3:
                    if geom_id in sel_secondary_geom_ids:
                        continue

                pts = geom[0][0]
                tooltip = None
                if geom_id > -1:
                    tooltip = folium.Tooltip(f'ID {geom_id}', sticky=True)

                geoms_to_add = []
                if self._geom_type == 'Arc':
                    geoms_to_add.append(folium.PolyLine(locations=pts, color=self._display_colors[type],
                                                        weight=weights[type], tooltip=tooltip,
                                                        dashArray=dash[type]))
                else:
                    geoms_to_add.append(folium.Polygon(locations=pts, color=self._display_colors[type],
                                                       weight=weights[type], tooltip=tooltip,
                                                       dashArray=dash[type], fill_color=fill_colors[type],
                                                       fill_opacity=opacity[type]))
                    if type != 3:
                        # don't draw holes of secondary polygons
                        for hole_idx in range(1, len(geom[0])):
                            geoms_to_add.append(folium.Polygon(locations=geom[0][hole_idx],
                                                               color=self._display_colors[type], weight=weights[type],
                                                               dashArray=dash[type]))

                if (type == 0 or type == 4) and self._draw_preview is True:
                    # this is the action geometry - only draw to action view
                    for drawing_item in geoms_to_add:
                        action_geoms.append(drawing_item)
                elif type == 1:
                    # this is the original geometry - only draw to map view
                    for drawing_item in geoms_to_add:
                        map_geoms.append(drawing_item)
                elif type == 2:
                    for drawing_item in geoms_to_add:
                        if self._draw_sel_secondary_for_action_preview and self._draw_preview is True:
                            action_geoms.append(drawing_item)
                        if self._update_map_view:
                            map_geoms.append(drawing_item)
                elif type == 5 and self._draw_preview is True:
                    for drawing_item in geoms_to_add:
                        action_geoms.append(drawing_item)
                else:
                    for drawing_item in geoms_to_add:
                        if self._draw_preview is True:
                            if geom_id not in action_ids_to_ignore:
                                action_geoms.append(drawing_item)
                        if self._update_map_view:
                            map_geoms.append(drawing_item)

        #  create the action map
        if self._draw_preview:
            if self._action_display_params:
                self._f_action_map = folium.Map(tiles='', control_scale=True,
                                                location=[self._action_display_params.lat,
                                                          self._action_display_params.long],
                                                zoom_start=self._action_display_params.zoom)
            else:
                self._f_action_map = folium.Map(tiles='', control_scale=True)
                self._f_action_map.fit_bounds(self._action_preview_extents)

            if self.ui.chk_show_map.isChecked():
                url = 'http://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}'
                tl = folium.raster_layers.TileLayer(tiles=url, name='ESRI World Imagery', attr='ESRI', opacity=0.4)
                tl.add_to(self._f_action_map)

            for geom in action_geoms:
                geom.add_to(self._f_action_map)

            self._f_action_map.save(self._url_action_preview_file)
            action_preview_map_name = self._f_action_map.get_name()

            action_parameters = ParametersObject("parameters", action_preview_map_name, self)
            self._action_preview.page().add_object(action_parameters)
            self._action_preview.page().runJavaScript(
                f"var bounds = {action_preview_map_name}.getBounds();"
                "var center = bounds.getCenter(); "
                f"window.set_parameters(center.lat, center.lng, {action_preview_map_name}.getZoom(), "
                "bounds.getWest(), bounds.getEast(), bounds.getSouth(), bounds.getNorth());")

            self._action_preview.setUrl(QUrl.fromLocalFile(self._url_action_preview_file))
            self._action_preview.show()
            self._preview_is_displayed = True
        else:
            self._clear_preview_window()
            self._preview_is_displayed = False

        # do the map view if we're doing that
        if self._update_map_view:
            self._map_viewer.clear()
            if self._map_viewer_params:
                self._f_viewer_map = folium.Map(tiles='', control_scale=True,
                                                location=[self._map_viewer_params.lat,
                                                          self._map_viewer_params.long],
                                                zoom_start=self._map_viewer_params.zoom)
            else:
                self._f_viewer_map = folium.Map(tiles='', control_scale=True)
                self._f_viewer_map.fit_bounds(self._map_viewer_extents)

            if self.ui.chk_show_map.isChecked():
                url = 'http://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}'
                tl = folium.raster_layers.TileLayer(tiles=url, name='ESRI World Imagery', attr='ESRI', opacity=0.4)
                tl.add_to(self._f_viewer_map)

            for geom in map_geoms:
                geom.add_to(self._f_viewer_map)

            self._f_viewer_map.save(self._url_map_view_file)
            self._map_viewer.update_viewer(self._f_viewer_map, self._url_map_view_file)

        self._update_map_view = False

    def _update_num_connected_arcs(self, arc_idx):
        """Update node assignments arc props."""
        if self._geom_type == 'Arc':
            ends = gsh.endpt_locs_from_ls(self._geoms[arc_idx])
            if ends[0] == ends[-1]:
                num = len(self._node_pts_to_arc_lst_idxs[ends[0]])
                if num == 1:
                    cx = -1
                else:
                    cx = num - 1
            else:
                cx = len(self._node_pts_to_arc_lst_idxs[ends[0]]) + len(self._node_pts_to_arc_lst_idxs[ends[1]]) - 2
            self._update_prop_for_geom_idx(arc_idx, STR_NUM_CX, cx)

    def _get_geom_feature_id(self, geom_idx):
        """Return the feature id that will be seen in SMS.

        Args:
            geom_idx (int): Geometry list index
        Returns:
            feature_id (int): Feature id.
        """
        return self._geom_df['feature_id'].iat[geom_idx]

    def _get_all_connected_polys(self, poly_idx):
        """Return all poly_idxs that use the arc.

        Args:
            poly_idx (int): Poly index
        Returns:
            all_polys (set): Set of poly idxs that are conencted.
        """
        to_check = set([poly_idx])
        all = set()

        while len(to_check) > 0:
            adj_polys = set()
            all.update(to_check)
            for check_me in to_check:
                adj_polys.update(self._get_adjacent_polys(check_me))
            all.update(to_check)
            to_check = {adj_poly for adj_poly in adj_polys if adj_poly not in all}
        return all

    def _get_adjacent_polys(self, poly_idx):
        """Return all poly_idxs that use the arc.

        Args:
            poly_idx (int): Poly index
        Returns:
            all_polys (set): Set of poly idxs that are adjacent to this poly.
        """
        poly_perim_arcs = self._get_poly_outer_arcs(poly_idx)
        all_polys = set()
        for arc in poly_perim_arcs:
            all_polys.update(self._get_polys_from_outer_arc(arc))

        if poly_idx in all_polys:
            all_polys.remove(poly_idx)

        return all_polys

    def _get_all_polys_using_arc(self, arc_idx):
        """Return all poly_idxs that use the arc.

        Args:
            arc_idx (int): Arc index
        Returns:
            all_polys (set): Set of poly idxs that use this arc.
        """
        all_polys = set(self._geom_df['a_outer_polys'].iat[arc_idx])
        all_polys.update(set(self._geom_df['a_inner_polys'].iat[arc_idx]))
        return all_polys

    def _get_all_arcs_from_poly(self, poly_idx):
        """Return all poly_idxs that use the arc.

        Args:
            poly_idx (int): Poly index
        Returns:
            all_arcs (set): Set of arcs idxs that are used by the poly.
        """
        all_arcs = set()
        inner_lists = self._get_poly_inner_arc_lists(poly_idx)

        # clean it up - if the poly shows up in two lists, two holes are adjacent to each other and that arc is
        # actually not part of the poly
        for list in inner_lists:
            for arc in list:
                if arc in all_arcs:
                    all_arcs.remove(arc)
                else:
                    all_arcs.add(arc)

        all_arcs.update(set(self._get_poly_outer_arcs(poly_idx)))
        return all_arcs

    def _get_poly_outer_arcs(self, poly_idx):
        """Get arcs that make up the outer boundary of the polygon.

        Args:
            poly_idx (int): Poly index
        Returns:
            all_arcs (set): Set of arcs idxs that are used by the poly for the perimeter.
        """
        return self._geom_df['p_outer_arcs'].iat[poly_idx].copy()

    def _set_poly_outer_arcs(self, poly_idx, outer_arcs, tmp_geom_dict=None):
        """Set the arcs that make up the outer boundary of the polygon.

        Args:
            poly_idx (int): Poly index
            outer_arcs (list): List of arcs that make up the perimeter
            tmp_geom_dict (dict): Temporary dictionary if we are potentially adding several geometry items at once
        """
        was_in_tmp = False
        if tmp_geom_dict:
            # check the temporary first
            if poly_idx in tmp_geom_dict['geom_idx']:
                dict_idx = tmp_geom_dict['geom_idx'].index(poly_idx)
                tmp_geom_dict['p_outer_arcs'][dict_idx] = outer_arcs
                was_in_tmp = True
        if not was_in_tmp:
            self._geom_df.at[poly_idx, 'p_outer_arcs'] = outer_arcs

        for outer_arc in outer_arcs:
            self._assign_outer_poly_to_arc(outer_arc, poly_idx, tmp_geom_dict)

    def _get_poly_inner_arc_lists(self, poly_idx):
        """Get a list of lists of arcs that make up holes in the poly - one list per hole.

        Args:
            poly_idx (int): Poly index
        Returns:
            (list): List of arc lists for each hole.
        """
        return self._geom_df['p_inner_arcs'].iat[poly_idx]

    def _set_poly_inner_arc_lists(self, poly_idx, inner_arc_lists, tmp_geom_dict=None):
        """Set list of lists of arcs that make up holes in the polygon.

        Args:
            poly_idx (int): Poly index
            inner_arc_lists (list): List of lists of arcs - one for each hole
            tmp_geom_dict (dict): Temporary dictionary if we are potentially adding several geometry items at once
        """
        was_in_tmp_geom = False
        if tmp_geom_dict:
            # check the temporary first
            if poly_idx in tmp_geom_dict['geom_idx']:
                dict_idx = tmp_geom_dict['geom_idx'].index(poly_idx)
                tmp_geom_dict['p_inner_arcs'][dict_idx] = inner_arc_lists
                was_in_tmp_geom = True
        if not was_in_tmp_geom:
            self._geom_df.at[poly_idx, 'p_inner_arcs'] = inner_arc_lists

        for inner_list in inner_arc_lists:
            candidate_polys = []
            for i, inner_arc in enumerate(inner_list):
                polys_from_arc = self._get_polys_from_outer_arc(inner_arc, tmp_geom_dict)
                if i == 0:
                    candidate_polys = polys_from_arc.copy()
                else:
                    candidate_polys = [poly for poly in polys_from_arc if poly in candidate_polys]

                if len(candidate_polys) == 1:
                    if was_in_tmp_geom:
                        tmp_geom_dict['p_containing_poly'][dict_idx] = candidate_polys[0]
                    else:
                        self._set_containing_poly(candidate_polys[0], poly_idx)

    def _get_polys_from_outer_arc(self, arc_idx, tmp_geom_dict=None):
        """Get polygons that use this arc as an outer arc.

        Args:
            arc_idx (int): Arc index
            tmp_geom_dict (dict): Temporary dictionary if we are adding multiple things at a time
        Returns:
            (list): List of polys that use this for perimeter.
        """
        if tmp_geom_dict:
            if arc_idx in tmp_geom_dict['geom_idx']:
                dict_idx = tmp_geom_dict['geom_idx'].index(arc_idx)
                return tmp_geom_dict['a_outer_polys'][dict_idx]
        return self._geom_df['a_outer_polys'].iat[arc_idx]

    def _get_polys_from_inner_arc(self, arc_idx):
        """Get polygons that use this arc as an inner arc.

        Args:
            arc_idx (int): Arc index
        Returns:
            (list): List of polys that use this for an inner arc.
        """
        return self._geom_df['a_inner_polys'].iat[arc_idx].copy()

    def _remove_outer_arc_from_poly(self, arc_idx, poly_idx):
        """Remove this arc from the list of arcs that make up the outer boundary of the polygon.

        Args:
            arc_idx (int): Arc index
            poly_idx (int): Poly index
        """
        outer_arcs = self._get_poly_outer_arcs(poly_idx)
        if arc_idx in outer_arcs:
            outer_arcs.remove(arc_idx)
        self._set_poly_outer_arcs(poly_idx, outer_arcs)

    def _remove_inner_arc_from_poly(self, arc_idx, poly_idx):
        """Remove this arc from the list of lists of arcs that are used for holes.

        Args:
            arc_idx (int): Arc index
            poly_idx (int): Poly index
            remove_entire_hole (bool): True if the hole using this arc should be deleted.
        """
        inner_arc_lists = self._get_poly_inner_arc_lists(poly_idx)
        hole_idxs = self._get_hole_polys(poly_idx)
        new_inner_arc_lists = []
        new_hole_idxs = []
        for i, inner_arc_list in enumerate(inner_arc_lists):
            if arc_idx not in inner_arc_list:
                new_inner_arc_lists.append(inner_arc_list)
                new_hole_idxs.append(hole_idxs[i])

        self._set_poly_inner_arc_lists(poly_idx, new_inner_arc_lists)
        self._set_poly_hole_polys(poly_idx, new_hole_idxs)

    def _remove_outer_poly_from_arc(self, arc_idx, poly_idx):
        """Unassign the polygon as one that uses this arc for its boundary.

        Args:
            arc_idx (int): Arc index
            poly_idx (int): Poly index
        """
        outer_polys = self._get_polys_from_outer_arc(arc_idx)
        if poly_idx in outer_polys:
            outer_polys.remove(poly_idx)
        # self._geom_df.at[arc_idx, 'a_outer_polys'] = outer_polys

    def _remove_inner_poly_from_arc(self, arc_idx, poly_idx):
        """Unassign the polygon as one that uses this arc for a hole.

        Args:
            arc_idx (int): Arc index
            poly_idx (int): Poly index
        """
        inner_polys = self._get_polys_from_inner_arc(arc_idx)
        if poly_idx in inner_polys:
            inner_polys.remove(poly_idx)
        self._geom_df.at[arc_idx, 'a_inner_polys'] = inner_polys

    def _get_hole_polys(self, poly_idx):
        """Get polygons that make up holes in the passed in polygon.

        Args:
            poly_idx (int): Poly index
        Returns:
            (list): List of polygon idxs that are holes of the passed in polygon
        """
        if poly_idx is not None:
            return self._geom_df['p_hole_polys'].iat[poly_idx].copy()
        return []

    def _remove_hole_from_poly(self, poly_idx, hole_idx):
        """Remove the hole polygon as one that is used in the specified containing poly.

        Args:
            poly_idx (int): Poly index
            hole_idx (int): Poly index of hole that is to be removed
        """
        hole_polys = self._get_hole_polys(poly_idx)
        hole_arc_lists = self._get_poly_inner_arc_lists(poly_idx)
        if hole_idx in hole_polys:
            i = hole_polys.index(hole_idx)
            hole_polys.remove(hole_idx)
            hole_arc_lists.remove(hole_arc_lists[i])
            self._set_poly_hole_polys(poly_idx, hole_polys)
            self._set_poly_inner_arc_lists(poly_idx, hole_arc_lists)

    def _remove_containing_poly_from_poly(self, hole_idx, containing_idx):
        """Remove the outer polygon as one that contains the hole polygon.

        Args:
            hole_idx (int): Poly index
            containing_idx (int): Poly index poly that contains hole_idx as a hole
        """
        if self._geom_df.at[hole_idx, 'p_containing_poly'] == containing_idx:
            self._geom_df.at[hole_idx, 'p_containing_poly'] = None

    def _assign_outer_poly_to_arc(self, arc_idx, outer_poly_idx, tmp_geom_dict=None):
        """Link the polygon as one that uses the specified arc in its boundary.

        Args:
            arc_idx (int): Arc index
            outer_poly_idx (int): Poly that uses arc_idx as in its perimeter
            tmp_geom_dict (dict): Temporary dictionary if we are potentially adding several geometry items at once
        """
        if tmp_geom_dict:
            # check the temporary first
            if arc_idx in tmp_geom_dict['geom_idx']:
                dict_idx = tmp_geom_dict['geom_idx'].index(arc_idx)
                outer_polys = tmp_geom_dict['a_outer_polys'][dict_idx]
                if outer_poly_idx not in outer_polys:
                    outer_polys.append(outer_poly_idx)
                tmp_geom_dict['a_outer_polys'][dict_idx] = outer_polys
                return

        outer_polys = self._get_polys_from_outer_arc(arc_idx)
        if outer_poly_idx not in outer_polys:
            outer_polys.append(outer_poly_idx)
        self._geom_df.at[arc_idx, 'a_outer_polys'] = outer_polys

    def _assign_inner_poly_to_arc(self, arc_idx, inner_poly_idx):
        """Link the polygon as one that uses the specified arc in a hole.

        Args:
            arc_idx (int): Arc index
            inner_poly_idx (int): Poly that uses arc_idx as in a hole
        """
        inner_polys = self._get_polys_from_inner_arc(arc_idx)
        if inner_poly_idx not in inner_polys:
            inner_polys.append(inner_poly_idx)
        self._geom_df.at[arc_idx, 'a_inner_polys'] = inner_polys

    def _set_poly_hole_polys(self, poly_idx, hole_poly_idxs):
        """Set the poly idxs that are used as holes.

        Args:
            poly_idx (int): Poly index of the containing poly
            hole_poly_idxs (list): List of polys that are holes in poly_idx
        """
        self._geom_df.at[poly_idx, 'p_hole_polys'] = hole_poly_idxs

    def _get_containing_poly(self, poly_idx):
        """Get the polygon that uses poly_idx as a hole.

        Args:
            poly_idx (int): Poly index of the hole
        Returns:
            (int): Index of the polygon that uses poly_idx as a hole, or None
        """
        c_poly = self._geom_df['p_containing_poly'].iat[poly_idx]
        if c_poly is not None:
            c_poly = int(c_poly)
        return c_poly

    def _set_containing_poly(self, poly_idx, containing_poly_idx):
        """Set the polygon that uses poly_idx as a hole.

        Args:
            poly_idx (int): Poly index of the hole
            containing_poly_idx (int): Poly index of the polygon that uses poly_idx as a hole
        """
        self._geom_df.at[poly_idx, 'p_containing_poly'] = containing_poly_idx

    def _assign_hole_to_containing_poly(self, containing_poly_idx, hole_poly_idx):
        """Set the polygon that uses poly_idx as a hole.

        Args:
            containing_poly_idx (int): Poly index of the polygon that uses poly_idx as a hole
            hole_poly_idx (int): Poly index of the hole
        """
        holes = self._get_hole_polys(containing_poly_idx)
        hole_arc_lists = self._get_poly_inner_arc_lists(containing_poly_idx)
        if hole_poly_idx not in holes:
            holes.append(hole_poly_idx)
            hole_arc_lists.append(self._get_poly_outer_arcs(hole_poly_idx))
            self._geom_df.at[containing_poly_idx, 'p_hole_polys'] = holes
            self._geom_df.at[containing_poly_idx, 'p_inner_arcs'] = hole_arc_lists

    """
    Qt overloads
    """
    def sizeHint(self):  # noqa: N802
        """Overridden method to help the dialog have a good minimum size.

        Returns:
            (:obj:`QSize`): Size to use for the initial dialog size.
        """
        return QSize(800, 500)

    def exec(self):
        """Overload to abort bringing up the dialog if we failed to load the tool output DataFrame."""
        if self.tool.abort:
            self.reject()
            return

        if self._process_data():
            return self._super_exec()

    def _process_data(self):
        """Overload to abort bringing up the dialog if we failed to load the tool output DataFrame."""
        dfs = load_tool_output()

        if dfs is None or len(dfs) != 2:
            return False

        self._ss_df = dfs[0]
        self._ss_df = self._ss_df.sort_values(STR_ID)
        self._secondary_ss_df = dfs[0]
        self._geom_df = dfs[1]

        self._new_geom_idx = len(self._geom_df)
        self._map_viewer = CleanShapesViewerDialog(parent=self)

        if self._show_viewer:
            self._map_viewer.show()

        self.ui = Ui_CleanShapesDialog()
        self._setup_ui()

        self._geoms = self.tool._geometry
        self._geom_type = 'Arc' if self.tool._geom_type == 'Arcs' else 'Polygon'
        self._geoms_for_props = self.tool._geometry_for_props

        self._simplify_tol = self.tool._simplify_tol

        # projection stuff
        self._native_wkt = self.tool._native_wkt
        self._geographic_wkt = self.tool._geographic_wkt
        self._non_geographic_wkt = self.tool._non_geographic_wkt
        self._coord_sys = self.tool._coord_sys
        self._is_meters = self.tool._horiz_units == 'METERS'
        self._is_local = self._coord_sys in ['NONE', '']
        self._is_geo = self._coord_sys == 'GEOGRAPHIC'
        self._convert = meters_to_decimal_degrees if self._is_meters else feet_to_decimal_degrees

        self._node_pts_to_arc_lst_idxs = self.tool._node_locs_to_arc_idxs
        self._rtree = self.tool._rtree

        if self._geom_type == 'Polygon':
            # finish mapping polygons
            all_outer = set()
            all_inner = set()
            for poly_idx in range(self.tool._num_arcs, len(self._geom_df)):
                outer_arcs = self._get_poly_outer_arcs(poly_idx)
                for arc in outer_arcs:
                    self._assign_outer_poly_to_arc(arc, poly_idx)
                all_outer.update(set(outer_arcs))
                inner_arc_lists = self._get_poly_inner_arc_lists(poly_idx)

                # reset this because the list will get created when we assign the inner polygons
                self._set_poly_inner_arc_lists(poly_idx, [])

                for inner_arc_list in inner_arc_lists:
                    for arc in inner_arc_list:
                        self._assign_inner_poly_to_arc(arc, poly_idx)
                    all_inner.update(set(inner_arc_list))

            for inner_arc in all_inner:
                # polys that use this arc as an "outer" arc will be holes
                polys_that_are_holes = self._get_polys_from_outer_arc(inner_arc)
                polys_that_contain = self._get_polys_from_inner_arc(inner_arc)

                for hole_poly in polys_that_are_holes:
                    # make sure that none of the hole outer arcs are part of the c poly outer arcs
                    for containing_poly in polys_that_contain:
                        self._set_containing_poly(hole_poly, containing_poly)
                        self._assign_hole_to_containing_poly(containing_poly, hole_poly)

        self._transform_all_geom_to_latlon()

        self._set_geom_type()

        return True

    def accept(self):
        """Overload to abort bringing up the dialog if we failed to load the tool output DataFrame."""
        # create the coverage
        self._create_cov()
        self._map_viewer.close()
        return super().accept()

    def _create_cov(self):
        """Create the coverage."""
        coverage_builder = CoverageBuilder(self.tool.query.display_projection.well_known_text,
                                           self.tool._output_coverage_name)

        # if there were disjoint points in the original coverage, add them
        for pt in self.tool._disjoint_points.itertuples():
            coverage_builder.add_point((pt.geometry.x, pt.geometry.y, pt.geometry.z))

        # create all of the arcs first
        list_id_to_new_arc = dict()
        for idx, geom in enumerate(self._geoms):
            if idx in self._deleted_geom_idxs or type(geom) is not LineString:
                continue

            pts = [(pt[0], pt[1], pt[2]) for pt in gsh.pts_from_ls(geom)]
            list_id_to_new_arc[idx] = coverage_builder.add_arc(pts, self._get_geom_feature_id(idx))

        # now create the polygons
        if self._geom_type == 'Polygon':
            for idx, geom in enumerate(self._geoms):
                if idx in self._deleted_geom_idxs or type(geom) is not shPolygon:
                    continue

                # only polygons get here
                poly_id = self._get_geom_feature_id(idx)
                arcs_old_ids = self._get_poly_outer_arcs(idx)

                # outer arcs
                outer = []
                points = []
                for old_id in arcs_old_ids:
                    if old_id not in self._deleted_geom_idxs:
                        new_id = list_id_to_new_arc[old_id]
                        geom = coverage_builder.get_arc_geometry(new_id)
                        if geom:
                            points.append(list(geom.coords))
                            outer.append(new_id)

                # interior arcs - if they are adjacent, we need them merged so that we only set the interior
                # to be what touches the outside polygon
                holes_to_process = self._get_hole_polys(idx)
                new_poly_inner_arcs = []
                holes = []
                while len(holes_to_process) > 0:
                    process_me = holes_to_process[0]
                    interior_group = self._get_all_connected_polys(process_me)

                    if len(interior_group) > 1:
                        # create a union of these adjacent holes so that we can get the outer perimeter of all
                        # of them - it messes it up we put in the arcs that they share, so this gets rid of them
                        all_involved_arcs = set()
                        overlapping_arcs = set()
                        for int_poly in interior_group:
                            poly_arcs = self._get_poly_outer_arcs(int_poly)
                            overlapping_arcs.update({arc for arc in poly_arcs if arc in all_involved_arcs})
                            all_involved_arcs.update(poly_arcs)

                        arcs_to_use = set([arc for arc in all_involved_arcs if arc not in overlapping_arcs])
                        if len(arcs_to_use) < 4:
                            int_poly_arc_list = [arc for arc in arcs_to_use]
                        else:
                            int_poly_arc_list = []
                            perim_pts = gsh.perim_pts_from_sh_poly_tuple(
                                unary_union([self._geoms[merge_idx] for merge_idx in interior_group]))
                            nodes_to_arc = dict()
                            for arc in arcs_to_use:
                                nodes = gsh.endpt_locs_from_ls(self._geoms[arc])
                                if nodes[0] not in nodes_to_arc.keys():
                                    nodes_to_arc[nodes[0]] = set()
                                nodes_to_arc[nodes[0]].add(arc)
                                if nodes[1] not in nodes_to_arc.keys():
                                    nodes_to_arc[nodes[1]] = set()
                                nodes_to_arc[nodes[1]].add(arc)

                            all_nodes = nodes_to_arc.keys()
                            node_locs = [pt for pt in perim_pts if pt in all_nodes]
                            arcs_at_node_1 = nodes_to_arc[node_locs[0]]
                            for i in range(len(node_locs) - 1):
                                arcs_at_node_2 = nodes_to_arc[node_locs[i + 1]]
                                shared_arcs = arcs_at_node_1.intersection(arcs_at_node_2)

                                the_arc = shared_arcs.pop()
                                int_poly_arc_list.append(the_arc)

                                if the_arc in nodes_to_arc[node_locs[i]]:
                                    nodes_to_arc[node_locs[i]].remove(the_arc)
                                if the_arc in nodes_to_arc[node_locs[i + 1]]:
                                    nodes_to_arc[node_locs[i + 1]].remove(the_arc)
                                if the_arc in arcs_to_use:
                                    arcs_to_use.remove(the_arc)
                                arcs_at_node_1 = arcs_at_node_2

                            if len(arcs_to_use) == 1:
                                int_poly_arc_list.append(arcs_to_use.pop())
                    else:
                        int_poly_arc_list = self._get_poly_outer_arcs(process_me)

                    interior_poly_arc_ids = []
                    inner_ring = []
                    for old_id in int_poly_arc_list:
                        arc_id = list_id_to_new_arc[old_id]
                        geom = coverage_builder.get_arc_geometry(arc_id)
                        if geom:
                            inner_ring.append(list(geom.coords))
                            interior_poly_arc_ids.append(arc_id)
                    holes.append(inner_ring)
                    new_poly_inner_arcs.append(interior_poly_arc_ids)
                    holes_to_process = list(set(holes_to_process) - interior_group)
                coverage_builder.add_polygon(points, holes, outer, new_poly_inner_arcs, poly_id)

        new_cov = coverage_builder.build_coverage()
        self.tool.set_output_coverage(new_cov, self.tool._arguments[2])

    def reject(self):
        """Overload to abort bringing up the dialog if we failed to load the tool output DataFrame."""
        if self._map_viewer:
            self._map_viewer.close()
        return super().reject()
