"""Class to manage display for MaterialComponent."""
# 1. Standard python modules
import os
import shutil

# 2. Third party modules
from PySide2.QtGui import QColor

# 3. Aquaveo modules
from xms.components.display.display_options_io import (read_display_option_ids, read_display_options_from_json,
                                                       write_display_option_ids, write_display_options_to_json)
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.core.filesystem import filesystem
from xms.guipy.data.category_display_option import CategoryDisplayOption
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.tuflowfv.data.material_data import MaterialData
from xms.tuflowfv.gui.material_parent_dialog import TfMaterialParentDialog


__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"


MAT_INITIAL_ATT_ID_FILE = 'initial_mat_polys.ids'
MAT_INITIAL_COMP_ID_FILE = 'initial_mat_comp.ids'


class MaterialComponentDisplay:
    """Class to manage display for MaterialComponent."""

    def __init__(self, comp):
        """Initializes the base component class.

        Args:
            comp (MaterialComponent): The component to manage
        """
        self._comp = comp

    def ensure_display_options_exist(self):
        """Ensure the component display options json file exists."""
        comp_dir = os.path.dirname(self._comp.main_file)
        self._comp.disp_opts_files[0] = os.path.join(comp_dir, 'material_display_options.json')
        if not os.path.exists(self._comp.disp_opts_files[0]):
            # Copy over the unassigned material category id file.
            default_id_file = os.path.join(os.path.dirname(os.path.dirname(__file__)),
                                           'gui', 'resources', 'default_data',
                                           f'mat_{MaterialData.UNASSIGNED_MAT}.matid')
            filesystem.copyfile(default_id_file, os.path.join(comp_dir, os.path.basename(default_id_file)))
            # Read the default display options, and save ourselves a copy with a randomized UUID.
            categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            default_file = os.path.join(os.path.dirname(os.path.dirname(__file__)),
                                        'gui', 'resources', 'default_data', 'material_default_display_options.json')
            json_dict = read_display_options_from_json(default_file)
            categories.from_dict(json_dict)
            json_dict['comp_uuid'] = os.path.basename(os.path.dirname(self._comp.main_file))
            write_display_options_to_json(self._comp.disp_opts_files[0], categories)
            # Save our display list UUID to the main file
            self._comp.data.info.attrs['display_uuid'] = categories.uuid
            self._comp.data.commit()

    def initialize_display(self, query):
        """Initialize the XMS component display list.

        Args:
            query (Query): The XMS interprocess communication object.

        Returns:
            tuple([], []): Empty message and ActionRequest lists
        """
        self._comp.cov_uuid = query.parent_item_uuid()
        if not self._comp.cov_uuid:
            return [('ERROR', 'Could not get TUFLOWFV material coverage UUID.')], []

        initial_att_file = os.path.join(os.path.dirname(self._comp.main_file), MAT_INITIAL_ATT_ID_FILE)
        if os.path.isfile(initial_att_file):  # Came from a model native read, initialize the component ids.
            att_ids = read_display_option_ids(initial_att_file)
            initial_comp_file = os.path.join(os.path.dirname(self._comp.main_file), MAT_INITIAL_COMP_ID_FILE)
            comp_ids = read_display_option_ids(initial_comp_file)
            filesystem.removefile(initial_att_file)
            filesystem.removefile(initial_comp_file)
            for att_id, comp_id in zip(att_ids, comp_ids):
                self._comp.update_component_id(TargetType.polygon, att_id, comp_id)

        self._comp.data.info.attrs['cov_uuid'] = self._comp.cov_uuid
        self._comp.data.commit()
        # Send the default material list to XMS.
        self._comp.display_option_list.append(
            XmsDisplayMessage(file=self._comp.disp_opts_files[0], edit_uuid=self._comp.cov_uuid)
        )
        return [], []

    def update_display_options_file(self):
        """Update the XMS display options JSON file to match what we have in memory."""
        category_list = CategoryDisplayOptionList()
        category_list.target_type = TargetType.polygon
        category_list.uuid = str(self._comp.data.info.attrs['display_uuid'])
        category_list.comp_uuid = self._comp.uuid
        df = self._comp.data.materials.to_dataframe()
        for mat_row in range(0, len(df.index)):
            category = CategoryDisplayOption()
            category.id = int(df.iloc[mat_row]['id'])
            category.description = df.iloc[mat_row]['name']
            category.file = f'mat_{category.id}.matid'
            category.options = PolygonOptions()
            style = df.iloc[mat_row]['color'].split()
            category.options.color = QColor(int(style[0]), int(style[1]), int(style[2]), 255)
            category.options.texture = int(style[3])
            # Make unassigned material the default category.
            if category.id == 0:  # Should always have 0 as "material id"
                category.is_unassigned_category = True
            category_list.categories.append(category)
        write_display_options_to_json(self._comp.disp_opts_files[0], category_list)

    def update_display_id_files(self, old_ids, new_ids):
        """Update the display files.

        Args:
            old_ids (list(int)): list of ids before editing materials
            new_ids (list(int)): list of current material ids

        Returns:
            list(int): deleted ids
        """
        deleted_ids = old_ids
        path = os.path.dirname(self._comp.main_file)
        for mat_id in new_ids:
            if mat_id >= 0:
                id_file = f'mat_{mat_id}.matid'
                filename = os.path.join(path, id_file)
                write_display_option_ids(filename, [mat_id])
            if mat_id in deleted_ids:
                deleted_ids.remove(mat_id)

        for mat_id in deleted_ids:
            id_file = f'mat_{mat_id}.matid'
            filename = os.path.join(path, id_file)
            filesystem.removefile(filename)
        return deleted_ids

    def update_material_list(self, query, old_ids):
        """Update the display options JSON file and display id files after editing the materials list.

        Args:
            query (Query): xmsapi interprocess communication object
            old_ids (list(int)): The old material ids
        """
        # Check for materials removed from the list.
        new_ids = list(self._comp.data.materials['id'].data.astype(int))
        deleted_ids = self.update_display_id_files(old_ids, new_ids)
        self.unassign_materials(query, deleted_ids)
        # Write the display options file.
        self.update_display_options_file()
        self._comp.display_option_list.append(
            XmsDisplayMessage(file=self._comp.disp_opts_files[0], edit_uuid=self._comp.cov_uuid)
        )

    def unassign_materials(self, query, delete_ids):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (Query): Object for communicating with XMS
            delete_ids (list(int)): List of the deleted material ids.

        Returns:
            Empty message and ActionRequest lists
        """
        query.load_component_ids(self._comp, polygons=True)
        if self._comp.cov_uuid in self._comp.comp_to_xms and \
                TargetType.polygon in self._comp.comp_to_xms[self._comp.cov_uuid]:
            poly_map = self._comp.comp_to_xms[self._comp.cov_uuid][TargetType.polygon]
            for mat in delete_ids:
                if mat in poly_map:
                    for att_id in poly_map[mat]:
                        self._comp.update_component_id(TargetType.polygon, att_id, MaterialData.UNASSIGNED_MAT)

    def open_assign_poly_materials(self, query, params, parent):
        """Opens the Assign Materials dialog and saves component data state on OK.

        Args:
            query (Query): Object for communicating with XMS
            params (dict): Generic map of parameters. Contains selection map and component id files.
            parent (QWidget): The Qt parent window container.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        # Get the XMS attribute ids of the selected polygons (if any)
        poly_ids = params.get('selection', [])
        num_polys = len(poly_ids)
        if num_polys == 0:
            return [('INFO', 'No polygons selected. Select one or more polygons to assign materials.')], []

        # Get the component id (material id) map of the selected polygons (if any).
        current_mat_id = MaterialData.UNASSIGNED_MAT
        id_files = params.get('id_files', [])
        if id_files is not None and len(id_files) > 1:
            self._comp.load_coverage_component_id_map({'POLYGON': (id_files[0], id_files[1])})
            # Delete the id dumped by xms files.
            if poly_ids:
                current_mat_id = self._comp.get_comp_id(TargetType.polygon, poly_ids[0])  # First if multiple selected.
                if current_mat_id == -1:
                    current_mat_id = MaterialData.UNASSIGNED_MAT

        current_mat_idx = MaterialData.UNASSIGNED_MAT
        for idx, mat_id in enumerate(self._comp.data.materials['id'].data):
            if mat_id == current_mat_id:
                current_mat_idx = idx
                break

        # Get original material ids, so we know if the user deletes one.
        old_ids = list(self._comp.data.materials['id'].data.astype(int))

        title = 'Assign Material' if num_polys == 1 else 'Assign Materials'
        dlg = TfMaterialParentDialog(title=title, parent=parent, material_data=self._comp.data,
                                     display_projection=query.display_projection, selected_material=current_mat_idx)
        if num_polys > 1:  # Add the multi-select warning if needed.
            dlg.add_multi_polygon_select_warning()

        if dlg.exec():
            self._comp.data = dlg.material_dialog.material_data
            self.update_material_list(query, old_ids)
            mat_id = int(self._comp.data.materials['id'].data.tolist()[dlg.selected_material])
            for poly_id in poly_ids:
                self._comp.update_component_id(TargetType.polygon, poly_id, mat_id)
            self._comp.data.commit()

        # Delete the id dumped by xms files.
        shutil.rmtree(os.path.join(os.path.dirname(self._comp.main_file), 'temp'), ignore_errors=True)
        return [], []

    def open_material_properties(self, query, parent):
        """Opens the Material Properties dialog and saves component data state on OK.

        Args:
            query (Query): Object for communicating with XMS
            parent (QWidget): The Qt parent window container.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        # get original material ids
        ids = self._comp.data.materials['id'].data.astype(int).tolist()
        dlg = TfMaterialParentDialog(title='Material List and Properties', parent=parent,
                                     material_data=self._comp.data, display_projection=query.display_projection)
        if dlg.exec():
            self._comp.data = dlg.material_dialog.material_data
            self.update_material_list(query, ids)
            self._comp.data.commit()
        return [], []
