"""Qt table view for defining global/gridded BCs."""
# 1. Standard python modules

# 2. Third party modules
import numpy as np
import pandas as pd
from PySide2.QtCore import QItemSelectionModel
from PySide2.QtWidgets import QAbstractItemView, QHeaderView

# 3. Aquaveo modules

# 4. Local modules
from xms.tuflowfv.data import bc_data as bcd
from xms.tuflowfv.gui import global_bcs_filter_model as gbfm
from xms.tuflowfv.gui.define_bc_button_delegate import DefineBcButtonDelegate, run_global_bc_dialog
from xms.tuflowfv.gui.tuflowfv_table import TuflowfvTableWidget


class GlobalBcsTableWidget(TuflowfvTableWidget):
    """The global/gridded BCs table widget."""

    def __init__(self, bc_data, parent, grid_id, time_formats):
        """Construct the widget.

        Args:
            bc_data (BcData): The model data
            parent (QDialog): The parent dialog
            grid_id (Union[int, None]): If gridded BC type, id of the grid whose datasets are being edited. Pass None
                if a non-gridded global BC.
            time_formats (tuple(str)): The current SMS user preference for formatting absolute datetimes. Should have
                specifiers for (strftime, qt)
        """
        self.bc_data = bc_data
        self.grid_id = grid_id
        self.time_formats = time_formats
        data = {'Type': self.bc_data.bcs.type.data, 'Attributes': np.full(len(self.bc_data.bcs.type.data), '')}
        self.data_frame = pd.DataFrame(data)
        self.data_frame.index += 1
        if self.grid_id is not None:
            defaults = {'Type': 'MSLP_GRID', 'Attributes': ''}
        else:
            defaults = {'Type': 'QG', 'Attributes': ''}
        super().__init__(parent=parent, data_frame=self.data_frame, select_col=gbfm.COL_IDX_DEFINE,
                         default_values=defaults)
        self.setup_ui()

    def setup_ui(self):
        """Add the table widget and initialize the model."""
        bc_location = bcd.BC_LOCATION_GRID if self.grid_id is not None else bcd.BC_LOCATION_GLOBAL
        define_delegate = DefineBcButtonDelegate(parent=self.parent(), bc_data=self.bc_data, bc_location=bc_location,
                                                 time_formats=self.time_formats)
        delegates = {gbfm.COL_IDX_DEFINE: define_delegate}
        filter_model = gbfm.GlobalBcsFilterModel(self)
        super()._setup_ui(column_delegates=delegates, stretch_last_section=True, fixed_size=False,
                          filter_model=filter_model)
        self.model.set_read_only_columns({gbfm.COL_IDX_TYPE})
        if bc_location == bcd.BC_LOCATION_GRID:
            self.table_view.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        else:  # I like interactive better, but it squeezes the first column too much in the add gridded BCs dialog.
            self.table_view.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)
        self.table_view.setSelectionMode(QAbstractItemView.SingleSelection)
        self.table_view.setSelectionBehavior(QAbstractItemView.SelectRows)

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        row_idx = self.model.rowCount()
        self.model.insertRows(row_idx, 1)
        new_index = self.model.index(row_idx, self.select_col)
        self.table_view.selectionModel().setCurrentIndex(
            new_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
        )
        bc_location = bcd.BC_LOCATION_GRID if self.grid_id is not None else bcd.BC_LOCATION_GLOBAL
        comp_id = self.bc_data.add_bc_atts(bc_location=bc_location)
        if bc_location == bcd.BC_LOCATION_GRID:
            self.bc_data.bcs['type'].loc[dict(comp_id=[comp_id])] = 'MSLP_GRID'
            self.bc_data.bcs['grid_id'].loc[dict(comp_id=[comp_id])] = self.grid_id
        else:
            self.bc_data.bcs['type'].loc[dict(comp_id=[comp_id])] = 'QG'
        run_global_bc_dialog(bc_data=self.bc_data, comp_id=comp_id, parent=self.parent(), bc_location=bc_location,
                             time_formats=self.time_formats)

    def on_btn_delete_row(self):
        """Called when a row is removed from the table."""
        indices = self.table_view.selectionModel().selectedIndexes()
        next_select_row = -1
        delete_rows = {index.row() if index.isValid() else -1 for index in indices}
        delete_count = 0
        delete_comp_ids = []
        for index in delete_rows:
            if index < 0:
                continue
            delete_row = index - delete_count
            next_select_row = delete_row - 1
            delete_comp_ids.append(self.bc_data.bcs.coords['comp_id'][index].item())
            self.model.removeRows(delete_row, 1)
            delete_count += 1
        self.bc_data.bcs = self.bc_data.bcs.where(~self.bc_data.bcs.comp_id.isin(delete_comp_ids), drop=True)
        if next_select_row >= 0:
            select_index = self.table_view.model().index(next_select_row, self.select_col)
            self.table_view.selectionModel().setCurrentIndex(
                select_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
            )
