"""Dialog to import TUFLOWFV CSV files."""
# 1. Standard python modules
import os
import re

# 2. Third party modules
from PySide2.QtGui import QIcon
from PySide2.QtWidgets import QComboBox, QFileDialog, QGridLayout, QLabel

# 3. Aquaveo modules
from xms.guipy import settings
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon

# 4. Local modules
from xms.tuflowfv.gui import gui_util


__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


class ImportCsvDialog(QFileDialog):
    """Dialog to import TUFLOWFV CSV files."""

    def __init__(self, columns, parent=None, start_dir=''):
        """Initializes the class, sets up the ui.

        Args:
            columns (dict): Key is default TUFLOWFV column name, value is label display text
            parent (Optional[QWidget]): The Qt parent, not provided if testing
            start_dir (Optional[str]): The start folder and potentially a initially selected file. If not provided
                will use the xmsguipy rules for determining an appropriate start directory.
        """
        self._start_dir = start_dir if os.path.exists(start_dir) else settings.get_file_browser_directory()
        super().__init__(parent=parent, caption='Import CSV', directory=start_dir, filter=gui_util.CSV_FILE_FILTER)
        self.delimiter = ',|\t'
        self._cbx_widgets = []
        self._setup_ui(columns)

    def _setup_ui(self, columns):
        """Add widgets to the dialog.

        Args:
            columns (dict): Key is default TUFLOWFV column name, value is label display text
        """
        self.setWindowIcon(QIcon(get_xms_icon()))
        self.setFileMode(QFileDialog.ExistingFile)  # Just open file dialog for now
        # Disable native dialog so we can add widgets
        opts = self.options()
        opts |= QFileDialog.DontUseNativeDialog
        self.setOptions(opts)
        # Add custom widgets to the layout for specifying column names
        main_layout = self.findChild(QGridLayout)
        for _, display_text in columns.items():
            row = main_layout.rowCount()
            label = QLabel(display_text)
            main_layout.addWidget(label, row, 0)
            cbx = QComboBox()
            main_layout.addWidget(cbx, row, 1)
            # Store the edit fields in order so we can return the column names in order
            self._cbx_widgets.append(cbx)
        self.currentChanged.connect(self.on_current_changed)
        if os.path.isfile(self._start_dir):  # If there is an initially selected file, populate the header comboboxes
            self.on_current_changed(self._start_dir)

    def on_current_changed(self, selection):
        """Slot called when selected file changes.

        Args:
            selection (str): The currently selected file or folder name
        """
        for combobox in self._cbx_widgets:
            combobox.setEnabled(False)
            combobox.clear()
        if not os.path.isfile(selection):
            return
        try:
            with open(selection, 'r') as f:
                header = f.readline().strip()
        except Exception:
            return

        header = re.split(self.delimiter, header)
        max_col_idx = len(header) - 1
        for idx, combobox in enumerate(self._cbx_widgets):
            combobox.setEnabled(True)
            for column in header:
                combobox.addItem(column)
            combobox.setCurrentIndex(min(idx, max_col_idx))

    def selected_file(self):
        """Returns the selected file, or empty string if user cancelled.

        Returns:
            str: See description
        """
        filenames = self.selectedFiles()
        return filenames[0] if filenames else ''

    def column_names(self):
        """Returns a list of the user specified column names.

        Returns:
            list[str]: See description
        """
        return [cbx.currentText() for cbx in self._cbx_widgets]

    def accept(self):
        """Save last used directory on OK."""
        filename = self.selected_file()
        if filename and os.path.isfile(filename):
            settings.save_file_browser_directory(os.path.dirname(filename))
        super().accept()
