"""QxTableView implementation."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtWidgets import QAbstractItemView

# 3. Aquaveo modules

# 4. Local modules
from xms.tuflowfv.data.sim_data import get_default_output_blocks_values, SUPPORTED_OUTPUT_TYPES
from xms.tuflowfv.gui.selected_combobox_delegate import SelectedComboboxDelegate
from xms.tuflowfv.gui.tuflowfv_table import TuflowfvTableWidget


__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


COL_OUTPUT_FORMAT = 0


class OutputBlocksTableWidget(TuflowfvTableWidget):
    """TuflowfvTableWidget implementation for use in the Model Control 'Output' tab."""

    def __init__(self, df, parent, data):
        """Initializes the class.

        Args:
            df (pandas.DataFrame): The data_frame to use in the table
            parent (QWidget): The Qt parent
            data (SimData): The simulation dataset
        """
        if not df.index.empty and df.index[0] == 0:
            df.index = df.index + 1  # Start index at 1, not 0
        super().__init__(parent=parent, data_frame=df, select_col=0,
                         default_values=get_default_output_blocks_values(False))
        self._data = data
        self._setup_ui()

    def _setup_ui(self, *args):
        """Setup the table."""
        cbx_delegate = SelectedComboboxDelegate(self)
        cbx_delegate.set_strings(SUPPORTED_OUTPUT_TYPES)
        delegates = {COL_OUTPUT_FORMAT: cbx_delegate}
        super()._setup_ui(column_delegates=delegates, stretch_last_section=True, fixed_size=False)
        # Hide all columns except the first
        for i in range(COL_OUTPUT_FORMAT + 1, self.model.columnCount()):
            self.table_view.setColumnHidden(i, True)
        # Don't require two clicks to expand combobox options
        triggers = QAbstractItemView.AllEditTriggers & ~QAbstractItemView.CurrentChanged
        self.table_view.setEditTriggers(triggers)
        # Hide the horizontal header. We only have one column and stretch it, so we essentially have a ListView.
        self.table_view.horizontalHeader().hide()
        # Select the entire row
        self.table_view.setSelectionMode(QAbstractItemView.SingleSelection)
        self.table_view.setSelectionBehavior(QAbstractItemView.SelectRows)

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        super().on_btn_add_row()
        row_idx = self.model.rowCount() - 1
        self.model.data_frame['row_index'].iloc[row_idx] = self._data.output.attrs['next_row_index']
        self._data.output.attrs['next_row_index'] += 1
