"""Qt table view for selecting tree items."""
# 1. Standard python modules

# 2. Third party modules
import numpy as np
from PySide2.QtWidgets import QAbstractItemView

# 3. Aquaveo modules

# 4. Local modules
from xms.tuflowfv.gui.tree_item_selector_delegate import (select_child_simulation, select_shapefile_or_coverage,
                                                          TreeItemSelectorDelegate)
from xms.tuflowfv.gui.tuflowfv_table import TuflowfvTableWidget


class TreeItemSelectorTableWidget(TuflowfvTableWidget):
    """The tree item selector table widget."""

    def __init__(self, tree_node, tree_type, data_frame, max_rows, parent, sim_uuid=None, layer_type=''):
        """Construct the widget.

        Args:
            tree_node (TreeNode): The root project explorer tree node
            tree_type (str): The target tree item type's XMS enum string, e.g. TI_GENERIC_POINT
            data_frame (pd.DataFrame): The model data
            max_rows (Union[int, None]): The maximum number of selected tree items
            parent (QDialog): The parent dialog
            sim_uuid (Optional[str]): UUID of the simulation selecting data. Only need to provide if
                tree_type==TI_DYN_SIM (selecting a linked child simulation).
            layer_type (Optional[str]): The type of the layer to select, e.g. 'Z', 'Output'
        """
        if len(data_frame.index) > 0:
            data_frame.index = np.arange(1, len(data_frame) + 1)
        # self._column = data_frame.columns[0]
        self._column = 0
        super().__init__(parent=parent, data_frame=data_frame, select_col=self._column, default_values={'uuid': ''})
        self._tree_node = tree_node
        self._tree_type = tree_type
        self._layer_type = layer_type
        self._max_rows = max_rows
        self._sim_uuid = sim_uuid
        self.setup_ui()

    def _update_add_button_state(self):
        """If there is a maximum number of rows, enable/disable the add row button when number of rows changes."""
        if self._max_rows is not None:
            enable = len(self.model.data_frame) < self._max_rows
            self.toolbar.widgetForAction(self.btn_actions[self.add_icon]).setEnabled(enable)

    def setup_ui(self):
        """Add the table widget and initialize the model."""
        selector_delegate = TreeItemSelectorDelegate(tree_type=self._tree_type, tree_node=self._tree_node,
                                                     parent=self.parent(), layer_type=self._layer_type,
                                                     sim_uuid=self._sim_uuid)
        delegates = {0: selector_delegate}
        super()._setup_ui(column_delegates=delegates, stretch_last_section=True, fixed_size=False)
        self.table_view.horizontalHeader().setVisible(False)
        self.table_view.setSelectionMode(QAbstractItemView.SingleSelection)
        self.table_view.setSelectionBehavior(QAbstractItemView.SelectRows)
        self._update_add_button_state()

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        if self._tree_type == 'TI_DYN_SIM':  # Select a child simulation
            item_uuid = select_child_simulation(sim_uuid=self._sim_uuid, tree_node=self._tree_node,
                                                previous_selection='', parent=self.parent())
        else:  # Select a Z line/point shapefile
            item_uuid = select_shapefile_or_coverage(tree_node=self._tree_node, previous_selection='',
                                                     tree_type=self._tree_type, layer_type=self._layer_type,
                                                     parent=self.parent())
        if item_uuid:
            super().on_btn_add_row()
            self.model.setData(self.model.index(self.model.rowCount() - 1, self._column), item_uuid)
        self._update_add_button_state()

    def on_btn_delete_row(self):
        """Called when a new row is added to the table."""
        super().on_btn_delete_row()
        self._update_add_button_state()
