"""Class to read a WaveWatch3 bounc (boundary input post-processing) namelist file."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import shlex
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules
from xms.wavewatch3.data.model import get_model
from xms.wavewatch3.file_io.io_util import READ_BUFFER_SIZE


class BouncNmlReader:
    """Class to read a WaveWatch3 bounc nml file."""
    def __init__(self, filename='', sim_data=None):
        """Constructor.

        Args:
            filename (:obj:`str`): Path to the nml file. If not provided (not testing or control file read),
                will retrieve from Query.
            sim_data(:obj:`xms.wavewatch3.data.SimData`):  The simulation data to edit.
        """
        self._filename = filename
        self._query = None
        self._sim_data = sim_data
        self._sim_comp_uuid = str(uuid.uuid4())
        self._setup_query()
        self._lines = []
        self._current_line = 0
        self._logger = logging.getLogger('xms.wavewatch3')
        self._global_values = get_model().global_parameters
        self._global_values.restore_values(self._sim_data.global_values)

    def _setup_query(self):
        """Setup the xmsapi Query for sending data to SMS and get the import filename."""
        if not self._filename:  # pragma: no cover - slow to setup Query for the filename
            self._query = Query()
            self._filename = self._query.read_file

    def _parse_next_line(self, shell=False):
        """Parse the next line of text from the file.

        Skips empty and comment lines.

        Args:
            shell (:obj:`bool`): If True will parse line using shlex. Slower but convenient for quoted tokens.

        Returns:
            (:obj:`list[str]`): The next line of text, split on whitespace
        """
        line = None
        while not line or line.startswith('!'):  # blank lines and control file identifier
            if self._current_line >= len(self._lines):
                # raise RuntimeError('Unexpected end of file.')
                return None
            line = self._lines[self._current_line].strip()
            self._current_line += 1
        if shell:
            return shlex.split(line, posix=False)
        return line.split()

    def _get_namelist_cards_and_values(self, firstline_data):
        """Reads the entire namelist until the closing / is found.  Stores cards and values.

        Handles cases where the data is on multiple lines, or on a single line.

        Args:
            firstline_data (:obj:`list[str]`):  List of data parsed on the opening line.

        Returns:
            (:obj:`dict`):  Dictionary with the keys being the cards read, and values of corresponding data.
        """
        namelist_list = firstline_data
        if '/' not in namelist_list:
            # We haven't found the end of the namelist yet.  Read more lines as necessary.
            end_not_found = True
            while end_not_found:
                # Grab the next line of data
                data = self._parse_next_line()
                # Extend the list of all data by the current line read
                namelist_list.extend(data)
                # Check if we've got the end of the namelist yet
                if '/' in data:
                    end_not_found = False

        # Now, we have the entire namelist as a list of values.  Parse into cards/commands and values
        cards = []
        values = []
        for i in range(len(namelist_list)):
            # Get the list elements on either side of each =
            if namelist_list[i] == '=':
                if 0 < i < len(namelist_list):
                    cards.append(namelist_list[i - 1].rstrip(','))
                    str = namelist_list[i + 1].rstrip(',')
                    if str[0] == "'" and str[-1] != "'":
                        # We have a string to read... get it until we find a closing single quote
                        # This could be something like a date:  '20150227 000000'
                        closing_found = False
                        while not closing_found:
                            i += 1
                            str += " " + namelist_list[i + 1].rstrip(',')
                            if str[-1] == "'":
                                closing_found = True
                    values.append(str)

        # Return the cards and values found throughout the namelist read
        return cards, values

    def _read_bounc_nml_file(self):
        """Read the BOUNC nml file."""
        reading = True
        while reading:
            data = self._parse_next_line()
            if data:
                if '&BOUND_NML' in data[0].strip():
                    self._read_bound_nml_namelist(data)
                else:
                    raise ValueError(f'Unrecognized namelist {data}')
            else:
                reading = False

    def _read_bound_nml_namelist(self, data):
        """Read the BOUND_NML namelist.

        Args:
            data(:obj:`list[str]`):  The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        run_control = model_parameters.group('run_control')
        self._logger.info('Reading BOUND_NML namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'BOUND%MODE' in cards:
            run_control.parameter('bound_mode').value = values[cards.index('BOUND%MODE')].lstrip("'").rstrip("'")
        if 'BOUND%INTERP' in cards:
            run_control.parameter('bound_interp').value = int(values[cards.index('BOUND%INTERP')])
        if 'BOUND%VERBOSE' in cards:
            run_control.parameter('bound_verbose').value = int(values[cards.index('BOUND%VERBOSE')])
        if 'BOUND%FILE' in cards:
            run_control.parameter('bound_file').value = values[cards.index('BOUND%FILE')].lstrip("'").rstrip("'")

    def read(self):
        """Top-level entry point for the WaveWatch3 bounc nml input file reader."""
        try:
            self._logger.info('Parsing ASCII text from file...')
            with open(self._filename, 'r', buffering=READ_BUFFER_SIZE) as f:
                self._lines = f.readlines()

            self._read_bounc_nml_file()
            self._logger.info('Committing changes....')
            self._sim_data.global_values = self._global_values.extract_values()
            self._sim_data.commit()
            self._logger.info('Finished!')
        except Exception:
            self._logger.exception(
                'Unexpected error in ounf nml preprocessor file '
                f'(line {self._current_line + 1}).'
            )
            raise
