"""Utilities for file IO operations."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import os

# 2. Third party modules
from PySide2.QtCore import QDateTime

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.guipy.time_format import ISO_DATETIME_FORMAT, qdatetime_to_datetime

# 4. Local modules

GEOGRAPHIC_WKT = 'GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101,' \
                 'AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,' \
                 'AUTHORITY["EPSG","8901"]],UNIT["degree",0.0174532925199433,AUTHORITY["EPSG","9122"]],' \
                 'AUTHORITY["EPSG","4269"]]'
LOCAL_METERS_WKT = 'LOCAL_CS["None",LOCAL_DATUM["None",0],UNIT["Meter",1],AXIS["None",OTHER]]'
READ_BUFFER_SIZE = 100 * 1024


def create_component_folder(comp_uuid):
    """Create a folder for a component in the XMS temp components folder.

    Args:
        comp_uuid (:obj:`str`): UUID of the new component

    Returns:
        (:obj:`str`): Path to the new component folder
    """
    temp_comp_dir = os.path.join(XmEnv.xms_environ_temp_directory(), 'Components', comp_uuid)
    os.makedirs(temp_comp_dir)
    return temp_comp_dir


def get_value_from_t_f(value):
    """Gets an integer (1 or 0) from the T or F string passed in.

    Args:
        value (:obj:`str`):  Value containing a T or F string.

    Returns:
        (:obj:`int`):  1 or 0 (for True of False) depending on the value passed in.
    """
    return 1 if value.upper() == 'T' else 0


def get_datetime_from_string(attrs, attr_name):
    """Get datetime from string in YYYY-MM-DD HH:MM:SS format.

    Args:
        attrs (:obj:`dict`): The attrs dict containing the datetime.
        attr_name (:obj:`str`): Key in attrs of the datetime.
    """
    try:  # Try to parse using current locale
        qreftime = QDateTime.fromString(attrs[attr_name])
        reftime = qdatetime_to_datetime(qreftime)
    except Exception:
        try:  # Try to parse using ISO format
            reftime = datetime.datetime.strptime(attrs[attr_name], ISO_DATETIME_FORMAT)
        except Exception:  # Set default to 01/01/1990
            reftime = datetime.datetime(1990, 1, 1)
    return reftime
